﻿//-----------------------------------------------------------------------
// <copyright file="SinglePointCalibrationSample.cs" company="KONICA MINOLTA, INC.">
//     (c) Copyright KONICA MINOLTA, INC.   All rights reserved.
// </copyright>
//-----------------------------------------------------------------------

using System;
using System.Collections.Generic;
using Konicaminolta; 

namespace SampleProgram
{
    class SinglePointCalibrationSample 
    {
        public static void Execute()
        {

            // Setup
            LightColorMISDK sdk = LightColorMISDK.GetInstance();
            ReturnMessage ret;

            // Connect
            ret = sdk.Connect();
            if (ret.errorCode != ErrorDefine.KmSuccess)
            {
                Console.WriteLine("Error in Connect():{0}", ret.errorCode);
                return;
            }

            // Set calibration channel 0
            ret = sdk.SetCalibrationCh(0);
            if (ret.errorCode != ErrorDefine.KmSuccess)
            {
                Console.WriteLine("Error in SetCalibrationCh():{0}", ret.errorCode);
                return;
            }

            // Start measurement (of a white point) before calibration
            ret = sdk.Measure();
            if( ret.errorCode != ErrorDefine.KmSuccess)
            {
                Console.WriteLine("Error in Measure(): {0}", ret.errorCode);
                return;
            }
            
            //  Polling status of measurement
            MeasStatus state;
            do
            {
                ret = sdk.PollingMeasurement(out state);
                if (ret.errorCode != ErrorDefine.KmSuccess)
                {
                    Console.WriteLine("Error in PollingMeasurement(): {0}", ret.errorCode);
                    return;
                }
            } while (state == MeasStatus.Measuring);
 
            // Read the value(Lv, x, y) before calibration
            Lvxy beforeValue     = new Lvxy(LuminanceUnit.cdm2);
            ret = sdk.ReadLatestData(beforeValue);
            if( ret.errorCode != ErrorDefine.KmSuccess)
            {
                Console.WriteLine("Error in ReadLatestData(): {0}", ret.errorCode);
                return;
            }

            // Prepare measurement values for calibration
            List<Lvxy> measValues     = new List<Lvxy>();
            measValues.Add(beforeValue);
            List<Lvxy> trueValues     = new List<Lvxy>();
            trueValues.Add( new Lvxy(LuminanceUnit.cdm2, 11.0, 0.4, 0.4) );

            // Set calibration with ID
            CalibType type      = CalibType.OnePoint;
            string id           = "some_id";    // you can set your favorite id
            int calibrationCh   = 1;
            ret = sdk.SetMatrixCalib(calibrationCh, measValues, trueValues, type, id);
            if (ret.errorCode != ErrorDefine.KmSuccess)
            {
                Console.WriteLine("Error in SetMatrixCalib():{0}", ret.errorCode);
                return;
            }

            //  Set calibration channel
            ret = sdk.SetCalibrationCh(calibrationCh);
            if (ret.errorCode != ErrorDefine.KmSuccess)
            {
                Console.WriteLine("Error in SetCalibrationCh():{0}", ret.errorCode);
                return;
            }

            //  Start measurement (of a white point) after calibration
            ret = sdk.Measure();
            if( ret.errorCode != ErrorDefine.KmSuccess)
            {
                Console.WriteLine("Error in Measure(): {0}", ret.errorCode);
                return;
            }


            //  Polling status of measurement
            do
            {
                ret = sdk.PollingMeasurement(out state);
                if (ret.errorCode != ErrorDefine.KmSuccess)
                {
                    Console.WriteLine("Error in PollingMeasurement(): {0}", ret.errorCode);
                    return;
                }
            } while (state == MeasStatus.Measuring);

            // Read the value(Lv, x, y) after calibration
            Lvxy afterValue = new Lvxy(LuminanceUnit.cdm2);
            ret = sdk.ReadLatestData(afterValue);
            if( ret.errorCode != ErrorDefine.KmSuccess)
            {
                Console.WriteLine("Error in ReadLatestData(): {0}", ret.errorCode);
                return;
            }

            // Display
            Console.WriteLine("Before calibration");
            foreach(var beforeCalibValue in beforeValue.ColorSpaceValue){
                Console.WriteLine("{0} : {1}", beforeCalibValue.Key, beforeCalibValue.Value);
            }
            Console.WriteLine("After calibration");
            foreach(var afterCalibValue in afterValue.ColorSpaceValue){
                Console.WriteLine("{0} : {1}", afterCalibValue.Key, afterCalibValue.Value);
            }

            //  Disconnect
            ret = sdk.DisConnect(0); 
        }
    }
}
