/** @brief      Sample code for a control of multiple instruments
                
    @file       AllMeasurement.cpp
    @author     KONICA MINOLTA, INC.

	Copyright(c) 2012-2016 KONICA MINOLTA, INC. All rights reserved.
*/

#include "../stdafx.h"
#include "../CLAPI.h"

void Execute();
bool RemoteOffCloseAll(DEVICE_HANDLE* handle, int32_km opened);
bool CloseAll(DEVICE_HANDLE* handle, int32_km opened);

int main(int argc, char* argv[])
{
	Execute();
	printf("Press any key.\n");
	getchar();
	return 0;
}

void Execute()
{
	const int32_km MAXDEVICE = 10;
	ER_CODE ret;

	// Acquire the device handle
	DEVICE_HANDLE handle[MAXDEVICE];
	bool bCalibEnd[MAXDEVICE] = {0};
	bool bMeasEnd[MAXDEVICE] = {0};
	int32_km i;
	int32_km opened = 0;

	for(i = 0; i < MAXDEVICE; i++){
		ret = CLOpenDevice(&handle[i]);
		if(ret == SUCCESS){
			ret = CLSetRemoteMode(handle[i], CL_RMODE_ON);
		}
		if(ret == SUCCESS){
			printf("Device[%d] : Connected.\n", i);
			opened++;
		}
	}

	if(opened == 0){
		printf("No Device\n");
		return;
	}

	// Perform a calibration
	// Start Calibration
	for(i = 0; i < opened; i++){
		printf("Device[%d] : Call CLDoCalibration()", i);
		ret = CLDoCalibration(handle[i]);
		if(ret != SUCCESS){
			printf("... Failed.\n");
			RemoteOffCloseAll(handle, opened);
			return;
		}
		printf("... Succeeded.\n");
	}

	// Polling Calibration
	CL_CALIBMEASSTATUS cstatus = CL_CALIBMEAS_FREE;
	for(i = 0; i < opened; i++){
		while(cstatus != CL_CALIBMEAS_FINISH){
			SLEEP(1000);
			printf("Device[%d] : Call CLPollingCalibration()", i);
			ER_CODE ret = CLPollingCalibration(handle[i], &cstatus);
			if(ret != SUCCESS){
				printf("... Failed.\n");
				RemoteOffCloseAll(handle, opened);
				return;
			}
			printf("... Succeeded.(status:%d)\n", cstatus);
		}
		// Clear the status to poll a next device
		cstatus = CL_CALIBMEAS_FREE;
	}

	for(i = 0; i < opened; i++){
		// Property setting(Observer)
		printf("Device[%d] : Call CLSetProperty(CL_PR_OBSERVER)", i);
		ret = CLSetProperty(handle[i], CL_PR_OBSERVER, CL_OBS_02DEG);
		if(ret != SUCCESS){
			printf("... Failed.\n");
			RemoteOffCloseAll(handle, opened);
			return;
		}
		printf("... Succeeded.\n");

		// Property setting(Illuminant unit)
		printf("Device[%d] : Call CLSetProperty(CL_PR_ILLUNIT)", i);
		ret = CLSetProperty(handle[i], CL_PR_ILLUNIT, CL_ILLUNIT_LX);
		if(ret != SUCCESS){
			printf("... Failed.\n");
			RemoteOffCloseAll(handle, opened);
			return;
		}
		printf("... Succeeded.\n");
	}

	// Perform a measurement
	printf("All Measurement 5seconds later.\n");
	SLEEP(5000);

	int32_km time;
	printf("Call CLDoMeasurementAll()");
	ret = CLDoMeasurementAll(&time);
	if(ret > WARNING){
		printf("... Failed.\n");
		RemoteOffCloseAll(handle, opened);
		return;
	}
	printf("... Succeeded.\n");
	
	//"time" is 100us Unit
	printf("Please wait for %dmm seconds.\n", time / 10);

	// Polling until completing a measurement
	CL_MEASSTATUS status = CL_MEAS_FREE;
	while(status != CL_MEAS_FINISH){
		SLEEP(1000);
		printf("Call CLPollingMeasureAll()");
		ER_CODE ret = CLPollingMeasureAll(&status);
		if(ret != SUCCESS){
			printf("... Failed.\n");
			RemoteOffCloseAll(handle, opened);
			return;
		}
		printf("... Succeeded.(status:%d)\n", status);
	}

	// Getting the measurement data(Ev/x/y)
	for(i = 0; i < opened; i++){
		CL_MEASDATA color;
		printf("Device[%d] : Call CLGetMeasData(CL_COLORSPACE_EVXY)", i);
		ret = CLGetMeasData(handle[i], CL_COLORSPACE_EVXY, &color);
		if(ret > WARNING){
			printf("... Failed.\n");
			RemoteOffCloseAll(handle, opened);
			return;
		}
		printf("... Succeeded\n");
		printf("[Ev]%f\n", color.Evxy.Ev);
		printf("[x ]%f\n", color.Evxy.x);
		printf("[y ]%f\n", color.Evxy.y);
	}
	// Post-processing
	RemoteOffCloseAll(handle, opened);
}

// Set remote mode off
bool RemoteOffCloseAll(DEVICE_HANDLE* handle, int32_km opened)
{
	ER_CODE ret;
	int32_km i;

	for(i = 0; i < opened; i++){
		printf("Device[%d] : Call CLSetRemoteMode(OFF)", i);
		ret = CLSetRemoteMode(handle[i], CL_RMODE_OFF);
		if(ret != SUCCESS){
			printf("... Failed.\n");
			return false;
		}
		printf("... Succeeded.\n");

		printf("Device[%d] : Call CLCloseDevice(OFF)", i);
		ret = CLCloseDevice(handle[i]);
		if(ret != SUCCESS){
			printf("... Failed.\n");
			return false;
		}
		printf("... Succeeded.\n");
	}
	return true;
}

// Release the device handle
bool CloseAll(DEVICE_HANDLE* handle, int32_km opened)
{
	ER_CODE ret;
	int32_km i;

	for(i = 0; i < opened; i++){ 
		printf("Device[%d] : Call CLCloseDevice", i);
		ret = CLCloseDevice(handle[i]);
		if(ret != SUCCESS){
			printf("... Failed.\n");
			return false;
		}
		printf("... Succeeded.\n");
	}
	return true;
}
