/** @brief      Sample code for a measurement
                
    @file       Measurement.cpp
    @author     KONICA MINOLTA, INC.

	Copyright(c) 2012-2016 KONICA MINOLTA, INC. All rights reserved.
*/

#include "../stdafx.h"
#include "../CLAPI.h"

void Execute();
bool RemoteOffClose(DEVICE_HANDLE handle);
bool Close(DEVICE_HANDLE handle);

int main(int argc, char* argv[])
{
	Execute();
	printf("Press any key.\n");
	getchar();
	return 0;
}

void Execute()
{
	ER_CODE ret;

	// Acquire the device handle
	DEVICE_HANDLE handle;

	printf("Call CLOpenDevice");
	ret = CLOpenDevice(&handle);
	if(ret != SUCCESS){
		printf("... Failed.\n");
		return;
	}
	printf("... Succeeded.\n");
	
	// Set remote mode on
	printf("Call CLSetRemoteMode(ON)");
	ret = CLSetRemoteMode(handle, CL_RMODE_ON);
	if(ret != SUCCESS){
		printf("... Failed .\n");
		Close(handle);
		return;
	}
	printf("... Succeeded.\n");

	// Perform a zero calibration
	printf("Call CLDoCalibration()");
	ret = CLDoCalibration(handle);
	if(ret != SUCCESS){
		printf("... Failed.\n");
		RemoteOffClose(handle);
		return;
	}
	printf("... Succeeded.\n");

	CL_CALIBMEASSTATUS cstatus = CL_CALIBMEAS_FREE;
	while(cstatus != CL_CALIBMEAS_FINISH){
		SLEEP(1000);
		printf("Call CLPollingCalibration()");
		ER_CODE ret = CLPollingCalibration(handle, &cstatus);
		if(ret != SUCCESS){
			printf("... Failed.\n");
			RemoteOffClose(handle);
			return;
		}
		printf("... Succeeded.(status:%d)\n", cstatus);
	}

	// Property setting(Observer)
	printf("Call CLSetProperty(CL_PR_OBSERVER)");
	ret = CLSetProperty(handle, CL_PR_OBSERVER, CL_OBS_02DEG);
	if(ret != SUCCESS){
		printf("... Failed.\n");
		RemoteOffClose(handle);
		return;
	}
	printf("... Succeeded.\n");

	// Property setting(Illuminant unit)
	printf("Call CLSetProperty(CL_PR_ILLUNIT)");
	ret = CLSetProperty(handle, CL_PR_ILLUNIT, CL_ILLUNIT_LX);
	if(ret != SUCCESS){
		printf("... Failed.\n");
		RemoteOffClose(handle);
		return;
	}
	printf("... Succeeded.\n");

	// Perform a measurement
	printf("Measurement 5seconds later.\n");
	SLEEP(5000);

	int32_km time;
	printf("Call CLDoMeasurement()");
	ret = CLDoMeasurement(handle, &time);
	if(ret > WARNING){
		printf("... Failed.\n");
		RemoteOffClose(handle);
		return;
	}
	printf("... Succeeded.\n");
	
	//"time" is 100us Unit
	printf("Please wait for %dmm seconds.\n", time / 10);

	// Polling until completing a measurement
	CL_MEASSTATUS status = CL_MEAS_FREE;
	while(status != CL_MEAS_FINISH){
		SLEEP(1000);
		printf("Call CLPollingMeasure()");
		ER_CODE ret = CLPollingMeasure(handle, &status);
		if(ret != SUCCESS){
			printf("... Failed.\n");
			RemoteOffClose(handle);
			return;
		}
		printf("... Succeeded.(status:%d)\n", status);
	}

	// Getting the measurement data(Ev/x/y)
	CL_MEASDATA color;
	printf("Call CLGetMeasData(CL_COLORSPACE_EVXY)");
	ret = CLGetMeasData(handle, CL_COLORSPACE_EVXY, &color);
	if(ret > WARNING){
		printf("... Failed.\n");
		RemoteOffClose(handle);
		return;
	}
	printf("... Succeeded\n");
	printf("[Ev]%f\n", color.Evxy.Ev);
	printf("[x ]%f\n", color.Evxy.x);
	printf("[y ]%f\n", color.Evxy.y);

	// Post-processing
	RemoteOffClose(handle);
}

// Set remote mode off
bool RemoteOffClose(DEVICE_HANDLE handle){
	ER_CODE ret;

	printf("Call CLSetRemoteMode(OFF)");
	ret = CLSetRemoteMode(handle, CL_RMODE_OFF);
	if(ret != SUCCESS){
		printf("... Failed.\n");
		return false;
	}
	printf("... Succeeded.\n");

	if(Close(handle) == false){
		return false;
	}

	return true;
}

// Release the device handle
bool Close(DEVICE_HANDLE handle){
	ER_CODE ret;

	printf("Call CLCloseDevice");
	ret = CLCloseDevice(handle);
	if(ret != SUCCESS){
		printf("... Failed.\n");
		return false;
	}
	printf("... Succeeded.\n");
	return true;
}
