/* Copyright (c) 2003 KONICA MINOLTA BUSINESS TECHNOLOGIES, INC. All rights reserved.	*/
/* TLI/Streams implementation of host software for TCP/IP communication option. 
 * The TLI functions are built on top of STREAMS and are available
 * to the programmer as a set of library routines.
 */


#include <stdio.h>
#include <fcntl.h>
#include <netdb.h>
#include <tiuser.h>
#include <sys/types.h>
#ifdef SOLARIS
#include <netinet/in.h>
#else
#include <sys/in.h>
#endif
#include <sys/socket.h>
#include <sys/errno.h>
#include <arpa/inet.h>
#include <stropts.h>
#include <poll.h>
#include <signal.h>
#include "qef.h"

char *index();
char *findstr();

struct sockaddr_in printaddr;
struct command_opt copts;

extern int t_errno;
char savebuf[MAX_REPLY_SIZE+1];
int postpagecnt=0;
extern errno;
extern short debug,
           nostat,
           bjobfile_present,
           blogfile_present,
           jobfile_present,
           logfile_present,
           verbose,
	   uflag,
           original;

FILE *bjobptr;
FILE *blogptr;
FILE *jobptr;
FILE *logptr;
struct ACCOUNT   account;
struct ACCOUNTING accounting;

main(argc, argv)
int argc;
char *argv[];
{
	int tpfd, ufdm, pstatus, sleeptime;
        char old_statstring[512], statbuf[512];
	struct sockaddr_in  claddr;
	struct hostent    *printer;
	struct file_list *flist;
	struct t_call *callptr;

	init_data(&copts);

	proc_args(argc, argv, &copts); 

        write_status(STAJSTART, "Starting job");
        if (!original){

             /*
              * If a log file was specified on the command line then
              * open the file in order to append the status messages returned
              */
             if (logfile_present) {
                 if (( logptr = fopen(copts.logfile, "a"))  ==  (FILE *)NULL) {
                         perror("Failed to open log file\n");
                         progexit(1);
                  }
             }

             /*
              * If a job file was specified on the command line then
              * open the file in order to write the status messages to it.
              */

             if (jobfile_present) {
                 if (( jobptr = fopen(copts.jobfile, "w"))  ==  (FILE *)NULL) {
                         perror("Failed to open job file\n");
                         progexit(1);
                  }
             }
        }

        /*
         * If a backchannel job file was specified on the command line then 
         * open the file in order to write the backchannel messages to it.
         */

        if (bjobfile_present && (strcmp(copts.bjobfile,copts.jobfile) != 0)) {
            if (( bjobptr = fopen(copts.bjobfile, "w"))  ==  (FILE *)NULL) {
                perror("Failed to open backchannel job file\n");
                progexit(1);
            }
        }
		else {
			if (bjobfile_present) {
				bjobptr = jobptr;
            }
        }

        /*
         * If a backchannel log file was specified on the command line then 
         * open the file in order to add the backchannel messages to it.
         */

        if (blogfile_present && (strcmp(copts.blogfile,copts.logfile) != 0)) {
            if (( blogptr = fopen(copts.blogfile, "a"))  ==  (FILE *)NULL) {
                perror("Failed to open backchannel log file\n");
                progexit(1);
            }
        }
		else {
	        if (blogfile_present) {
				blogptr = logptr;
            }
        }

	if ((printer = gethostbyname(copts.hostname)) == (struct hostent *)NULL) {
		fprintf(stderr, "gethostbyname failed:hostname not found in /etc/hosts\n");
		exit(1);
	}
#ifndef SOLARIS

	DBGPRT(("printer hostent fields:\n h_name= %s h_addr= %s h_length=0x%x\n", 
		printer->h_name, inet_ntoa((struct in_addr *)(printer->h_addr)), printer->h_length))

#endif
	/*
	 * Set up the address of the peer process with whom we want to
	 * communicate. The same address (printaddr) can be used for both
	 * TCP and UDP connections since both tcp port and udp port numbers
	 * are the same.
	 */

	jzero((char *)&printaddr, sizeof(printaddr)); 
	printaddr.sin_family = AF_INET;
	printaddr.sin_port = htons(copts.port_number);
 	jcopy((char *)printer->h_addr, (char *)&printaddr.sin_addr, printer->h_length);
	
	flist = copts.flist;
	while (copts.input_stdin || (flist != (struct file_list *)NULL)) {

	/*
	 * Set up UDP TLI for printer statuses.
	 */
	if ((ufdm = set_udptli()) < 0) {
		perror("Failed to open UDP TLI link for printer statuses\n");
		progexit(1);
	}

	DBGPRT(("Printer Status UDP file descriptor=%d\n", ufdm))


	/*
	 * Get printer status using UDP based STATUS1 protocol.
	 */
	pstatus = get_printstat(ufdm, 1, statbuf, 0);
	sleeptime = SLEEPTIME;
	while (pstatus == 0) {
		sleep(sleeptime);
		pstatus = get_printstat(ufdm, 1, statbuf, 0);
                if (!original)
                    process_message(statbuf, old_statstring);
	}
	
        /*
         * Spawn the child process for obtaining status1 info from the printer.
         */
        if (!nostat) {
            copts.cpid[0] = spawn_statproc();
        }

        /*
         * Spawn the child process for obtaining backchannel messages from
		 * the printer.
         */
        if (bjobfile_present || blogfile_present) {
            copts.cpid[1] = spawn_bchanproc();
        }

	/*
	 * Open TCP TLI.
	 */
	if ((tpfd = t_open(TCP_PROVIDER, O_RDWR, (struct t_info *)0)) < 0) {
                t_error("Failed to open connection to transport provider\n");
                progexit(1);
        }

	DBGPRT(("TCP file descriptor=%d\n", tpfd))
        /*
         * Since we are the 'active' partner in the communication, we don't
         * really care what address is bound to our transport endpoint, tpfd;
         * happy with whatever Transport Provider provides. Same holds true
         * for maximum number of connections.
         */

         if (t_bind(tpfd, (struct t_bind *)0, (struct t_bind *)0) < 0) {
                t_error("Failed to bind address to endpoint\n");
                progexit(1);
         }

	/*
     	 * Connect (TCP) to the peer on the printer.
	 */

	if ((callptr = (struct t_call *)t_alloc(tpfd, T_CALL, T_ADDR)) == NULL) {
                t_error("Failed to allocate t_call structure\n");
                progexit(1);
        }

        callptr->addr.maxlen = sizeof(printaddr);
        callptr->addr.len = sizeof(printaddr);
        callptr->addr.buf = (char *)&printaddr;
        callptr->opt.len = 0;
        callptr->udata.len = 0;


	if (t_connect(tpfd, callptr, (struct t_call *)0) < 0) {
		t_error("Failed to connect to printer\n");
		progexit(1);
	}

 	/*
	 * Printer is accepting data; Transfer it (over TCP).
	 */
	if (xfer_data(tpfd, &copts, flist) < 0) {
		printf("Data Transfer to the printer failed\n");
		write_status(STAACOMP, "communications ended abnormally");
		progexit(1);
	}

	if (t_close(tpfd) < 0) {
		t_error("Failed to close tcp descriptor\n");
		progexit(1);
	}
	
	if (t_close(ufdm) < 0) {
		t_error("Failed to close udp descriptor\n");
		progexit(1);
	}

	/*
	* Wait for the child to receive a terminating status
	* request from the printer. (if using accounting)
	*/
	if (!original) {
		while (wait(0) != copts.cpid[0]);
	}

	write_status(STAJCOMP, "Job transmission completed");

	if (original){

		/* Terminate the status gathering process */
		if (!nostat) {
			kill_statproc();
		}
	}

	/* Terminate the backchannel process */
	if (bjobfile_present || blogfile_present) {
		kill_bchanproc();
	}

	/* Disable stdin so that next loop fails. */
	if (copts.input_stdin) {
		copts.input_stdin = 0;
	}
	else {
		flist = flist->next;
	}

	} /* end while file list */

        if (!original){
             /*
              * Close the status files if they were open.
              */

             if (jobfile_present)
                 if (fclose(jobptr) < 0) {
                         perror("Failed to close status file\n");
                         progexit(1);
                 }
             if (logfile_present)
                 if (fclose(logptr) < 0) {
                         perror("Failed to close status file\n");
                         progexit(1);
                 }
        }

        /*
         * Close the backchannel files if they were open.
         */
        if (bjobfile_present && (strcmp(copts.bjobfile,copts.jobfile) != 0)) {
            if (fclose(bjobptr) < 0) {
                perror("Failed to close backchannel job file\n");
                progexit(1);
            }
        }
        if (blogfile_present && (strcmp(copts.blogfile,copts.logfile) != 0)) {
            if (fclose(blogptr) < 0) {
                perror("Failed to close backchannel log file\n");
                progexit(1);
            }
        }

        write_status(STASTERM, "Spooler terminated communication");
	    exit(0);
}

xfer_data(tpfd, copts, flist)
int tpfd;
struct command_opt *copts;
struct file_list *flist;
{
	int rfd, nw, nr, w, tobewritten;
	char buf[BUFSIZE], *bufp;

	if (copts->input_stdin) {
		rfd = 0;
	}
	else {
		if ((rfd = open(flist->filename, O_RDONLY, 0)) < 0) {
			perror("Open of input file failed\n");
			progexit(1);
		}
	}

	while (( nr = read(rfd, buf, BUFSIZE)) !=  0) {
		if (nr < 0) {
			perror("read of input file failed\n");
		    return(-1);	
		}	

		nw = 0;
		bufp = buf;
		tobewritten = nr;

		while (nw != nr) {
			if ((w = t_snd(tpfd, bufp, tobewritten, 0)) < 0) {
				t_error("send across tcp TLI failed\n");
			    return(-1);	
			}
			nw +=w;
			bufp += w;
			tobewritten -= w;
		}
	}
}

set_udptli()
{
	int ufd;

	if ((ufd = t_open(UDP_PROVIDER, O_RDWR, (struct t_info *)0)) < 0) {
                t_error("Failed to open connection to transport provider\n");
                progexit(1);
        }

        /*
         * Since we are the 'active' partner in the communication, we don't
         * really care what address is bound to our transport endpoint, tpfd; happy
         * with whatever Transport Provider provides. Same holds true for maximum
         * number of connections.
         */

        if (t_bind(ufd, (struct t_bind *)0, (struct t_bind *)0) < 0) {
                t_error("Failed to bind address to endpoint\n");
                progexit(1);
        }
	
	return(ufd);

}
	
init_backchannel(ufd)
int ufd;
{
	int nfound, nfd, rcnt, rflags, protocol_error, protocol_tries=1;
	char *strptr;
	struct printer_pkt  preq, *ps; 
	struct t_unitdata sudata, *rudata;
	struct t_uderr *ruderr;
	char  reply[MAX_REPLY_SIZE];
	struct pollfd fds[1];


	/* 
	 * Allocate a unitdata structure and its address field.
	 */
	if ((rudata = (struct t_unitdata *)t_alloc(ufd, T_UNITDATA, T_ADDR))
       		==  (struct t_unitdata *) NULL){
     	t_error("Failed to allocate Rcv unitdata\n");
       	progexit(1);
    }

	/* 
     * Allocate a uderr structure and its buffers.
     */
    if ((ruderr = (struct t_uderr *)t_alloc(ufd, T_UDERROR, T_ALL))
          ==  (struct t_uderr *) NULL){
        t_error("Failed to allocate uderror structure\n");
        progexit(1);
    }

	/* Flush out past reply packets, if any */
	protocol_error = FALSE;
    while (!protocol_error) {
	while (TRUE) {
		DBGPRT(("backchanel UFD = %d\n", ufd))

		fds[0].events = fds[0].revents = 0;
		fds[0].fd = ufd;
		fds[0].events = fds[0].events | POLLIN;

		nfd = 1;

		nfound = poll(fds, nfd, POLL_TIMEOUT);

		if ((nfound < 0) || (nfound > 1)) {
			perror("Poll failed\n");
			progexit(1); 
		}
		else if (nfound == 1) {
			DBGPRT(("Found an old reply packet\n"))

			rudata->opt.maxlen = 0;
			rudata->udata.maxlen = MAX_REPLY_SIZE;
			rudata->udata.buf = reply;

			    if (t_rcvudata(ufd, rudata, &rflags) < 0) { 
				    if (t_errno == TLOOK) {
               		    if (t_rcvuderr(ufd, ruderr) < 0) {
                   		    fprintf(stderr, "t_rcvuderr failed\n");
                   		    progexit(1);
              		    }
               		    fprintf(stderr, "Bad datagram: Error=%d\n", ruderr->error); 
              		    progexit(1);
           		    }
           		    else {
           			    t_error("Failed to receive udata\n");
               		    progexit(1);
          		        }
			        }
		        }
		        else {
			        DBGPRT(("Timed out: No[more] past reply packets\n"))
			        break;
		        }
	        }

	    rcnt = NUMRETRY;
	    while (rcnt--) {

		/*
	 	 * Initialize udata to be sent to the printer.
         */

     	    sudata.addr.maxlen = sizeof(printaddr);
     	    sudata.addr.len = sizeof(printaddr);
   	        sudata.addr.buf = (char *)&printaddr;

     	    sudata.opt.maxlen = 0;
     	    sudata.opt.len = 0;
     	    sudata.opt.buf = (char *)0;

     	    jzero((char *)&preq, sizeof(preq));
            if (bjobfile_present || blogfile_present) {
                preq.pr_spare = 2;
            }
            else {
                preq.pr_spare = 3;
            }

         	sudata.udata.maxlen = sizeof(preq);
         	sudata.udata.len = sizeof(preq);
         	sudata.udata.buf = (char *) &preq;


        	if (t_sndudata(ufd, &sudata) < 0) {
               		t_error("Failed to send udp data \n");
                	progexit(1);
        	}

		/*
		 * Issue a poll and see if you get a packet within the timeout interval.
		 */


		    DBGPRT(("backchanel UFD = %d\n", ufd))

		    fds[0].events = fds[0].revents = 0;
		    fds[0].fd = ufd;
		    fds[0].events = fds[0].events | POLLIN;

		    nfd = 1;

		    nfound = poll(fds, nfd, POLL_TIMEOUT);

		    if ((nfound < 0) || (nfound > 1)) {
			    perror("Poll failed\n");
			    progexit(1); 
		    }
		    else if (nfound == 0) {
			    DBGPRT(("Timed out: Reply packet not received from printer\n"))
			    continue; 
		        }
		        else {
			    DBGPRT(("Reply packet received from printer\n"))
		        }

		    if (fds[0].revents & POLLIN) { 
			    DBGPRT(("Poll:: Data available on UDP streams\n"))
		    }

 		    rudata->opt.maxlen = 0;
       	    rudata->udata.maxlen = MAX_REPLY_SIZE;
            rudata->udata.buf = reply;

            if (t_rcvudata(ufd, rudata, &rflags) < 0) {
        	    if (t_errno == TLOOK) {
                   	if (t_rcvuderr(ufd, ruderr) < 0) {
                   		 fprintf(stderr, "t_rcvuderr failed\n");
                   		 progexit(1);
                   	}
                   	fprintf(stderr, "Bad datagram: Error=%d\n", ruderr->error);
                   	progexit(1);
           	    }
           	    else {
           		    t_error("Failed to receive udata\n");
               	    progexit(1);
           	    }
            }

		
		    ps = (struct printer_pkt *)rudata->udata.buf;

		    DBGPRT(("Sent DB pr_spare=%d, received pr_spare=%d\n",preq.pr_spare,ps->pr_spare))
            if ((bjobfile_present || blogfile_present) && (ps->pr_spare == 2)) {
                protocol_error = TRUE;
            }
            else {
                if (ps->pr_spare == 3) {
                    protocol_error = TRUE;
                }
                else {
                    if (protocol_tries == 2) {
		                if (bjobfile_present) {
                            fprintf(bjobptr, "Your printer firmware does not support backchannel message retrieval!\n");
                            fprintf(bjobptr, "(or the printer keypad Accnting/RevChan Mode is Disabled)\n");
                            fflush(bjobptr);
                        }
		                if (blogfile_present) {
                            fprintf(blogptr, "Your printer firmware does not support backchannel message retrieval!\n");
                            fprintf(blogptr, "(or the printer keypad Accnting/RevChan Mode is Disabled)\n");
                            fflush(blogptr);
                        }

                        protocol_error =  TRUE;
                        bjobfile_present = blogfile_present = FALSE;
			DBGPRT(("DB does not support backchannel!\n"))
                    }
		            else {
			            protocol_tries++;
                    }
                }
            }
            if (protocol_error) {
                return(1);	
            }
        }
        if (protocol_error) {
            return(0);
        }
	}
}

/*
 * Get printer status. This is an infinite loop; if no replies
 * are being received from the printer, the procedure keeps on sending
 * requests in the hope that a reply will be received. It would be trivial,
 * however, to add logic to terminate the procedure if no replies are
 * received after a long timeout, say 5 minutes.
 * Use of select() enables low level timeout mechanism to be enforced and so,
 * in theory, a network read, i.e. recvfrom(), should never block.
 * A reply packet from the printer indicates if the printer is busy or not.
 * If it is, the procedure, after RETRY counts, sleeps for SLEEPTIME before
 * sending the request packet again. The priming select() outside the 
 * inner while loop (RETRY loop), checks to see if any old packets have
 * finally found their destination and discards them.
 */
get_printstat(ufd, statflag, statbuf, term)
int ufd;
int statflag;
char *statbuf;
int term;
{
	int nfound, nfd, rcnt, rflags, pstatus, recvlen, age;
	char *strptr;
	struct printer_pkt  preq, *ps; 
	struct t_unitdata sudata, *rudata;
	struct t_uderr *ruderr;
	char  reply[MAX_REPLY_SIZE];
	struct pollfd fds[1];


	/* 
	 * Allocate a unitdata structure and its address field.
	 */

	if ((rudata = (struct t_unitdata *)t_alloc(ufd, T_UNITDATA, T_ADDR))
           		==  (struct t_unitdata *) NULL){
               	t_error("Failed to allocate Rcv unitdata\n");
               	progexit(1);
        }

        /* Allocate a uderr structure and its buffers.
         */
        if ((ruderr = (struct t_uderr *)t_alloc(ufd, T_UDERROR, T_ALL))
              ==  (struct t_uderr *) NULL){
                t_error("Failed to allocate uderror structure\n");
                progexit(1);
        }

	/* Flush out past reply packets, if any */

	while (TRUE) {
		DBGPRT(("UFD = %d\n", ufd))

		fds[0].events = fds[0].revents = 0;
		fds[0].fd = ufd;
		fds[0].events = fds[0].events | POLLIN;

		nfd = 1;

		nfound = poll(fds, nfd, POLL_TIMEOUT);

		if ((nfound < 0) || (nfound > 1)) {
			perror("Poll failed\n");
			progexit(1); 
		}
		else if (nfound == 1) {
			DBGPRT(("Found an old reply packet\n"))

			rudata->opt.maxlen = 0;
			rudata->udata.maxlen = MAX_REPLY_SIZE;
			rudata->udata.buf = reply;

			if (t_rcvudata(ufd, rudata, &rflags) < 0) { 
				if (t_errno == TLOOK) {
                        		if (t_rcvuderr(ufd, ruderr) < 0) {
                                		fprintf(stderr, "t_rcvuderr failed\n");
                                		progexit(1);
                        		}
                        		fprintf(stderr, "Bad datagram: Error=%d\n", ruderr->error); 
                        		progexit(1);
                		}
                		else {
                       			t_error("Failed to receive udata\n");
                        		progexit(1);
                		}
			}
		}
		else {
			DBGPRT(("Timed out: No[more] past reply packets\n"))
			break;
		}
	}

	pstatus = 0;
	rcnt = NUMRETRY;
	while (rcnt--) {

		/*
	 	 * Initialize udata to be sent to the printer.
        	 */

         	sudata.addr.maxlen = sizeof(printaddr);
         	sudata.addr.len = sizeof(printaddr);
       	 	sudata.addr.buf = (char *)&printaddr;

         	sudata.opt.maxlen = 0;
         	sudata.opt.len = 0;
         	sudata.opt.buf = (char *)0;

         	jzero((char *)&preq, sizeof(preq));
                if (term)
                    preq.pr_spare = 1;
         	sudata.udata.maxlen = sizeof(preq);
         	sudata.udata.len = sizeof(preq);
         	sudata.udata.buf = (char *) &preq;


        	if (t_sndudata(ufd, &sudata) < 0) {
               		t_error("Failed to send udp data \n");
                	progexit(1);
        	}

		/*
		 * Issue a poll and see if you get a packet within the timeout interval.
		 */


		DBGPRT(("UFD = %d\n", ufd))

		fds[0].events = fds[0].revents = 0;
		fds[0].fd = ufd;
		fds[0].events = fds[0].events | POLLIN;

		nfd = 1;

		if ((nfound = poll(fds, nfd, POLL_TIMEOUT)) < 0) {
			perror("Poll failed\n");
			progexit(1);
		}

		if ((nfound < 0) || (nfound > 1)) {
			perror("Poll failed\n");
			progexit(1); 
		}
		else if (nfound == 0) {
			DBGPRT(("Timed out: Reply packet not received from printer\n"))
			continue; 
		}
		else {
			DBGPRT(("Reply packet received from printer\n"))
		}

		if (fds[0].revents & POLLIN) { 
			DBGPRT(("Poll:: Data available on UDP streams\n"))
		}

 		rudata->opt.maxlen = 0;
       		rudata->udata.maxlen = MAX_REPLY_SIZE;
        	rudata->udata.buf = reply;

        	if (t_rcvudata(ufd, rudata, &rflags) < 0) {
               		if (t_errno == TLOOK) {
                        	if (t_rcvuderr(ufd, ruderr) < 0) {
                               		 fprintf(stderr, "t_rcvuderr failed\n");
                               		 progexit(1);
                        	}
                        	fprintf(stderr, "Bad datagram: Error=%d\n", ruderr->error);
                        	progexit(1);
                	}
                	else {
                       		t_error("Failed to receive udata\n");
                        	progexit(1);
                	}
        	}

		
		ps = (struct printer_pkt *)rudata->udata.buf;

		DBGPRT(("Printer Status data:\n\t pr_status:0x%x\n\t pr_acceptmask:0x%x\n",
			ps->pr_status, ntohl(ps->pr_acceptmask)))

		pstatus = (!(ps->pr_status & 0x0f)) & (ntohl(ps->pr_acceptmask) & 0x01);
		DBGPRT(("print_status=%x\n", pstatus))

			
#ifdef DEBUG
		if (ps->pr_status & 0x02) 
			DBGPRT(("Printer lacks paper\n"))
			
		if (ps->pr_status & 0x04)
			DBGPRT(("Printer has paper jam\n"))
			
		if (ps->pr_status & 0x08)
			DBGPRT(("Printer lacks consumables (other than paper)\n"))

		if (ps->pr_status & 0x10)
			DBGPRT(("Job in progress at the printer\n"))

		if (pstatus != 0) {
			DBGPRT(("...printer ready to accept data\n"))
		}
		else {
			DBGPRT(("....printer busy or has problems\n"))
		}
#endif

		age = ntohl(ps->pr_age);
                /*
                 * Copy the time from the age packet into the Time field
                 * of the accounting structure.
                 */

                 sprintf(account.Time, "%02d:%02d:%02d",
                           age/3600, (age/60)%60, age%60);
                 if (ps->pr_status ==  0)
                        accounting.Terminate = 1;
                 else
                        accounting.Terminate = 0;



		strptr = (&rudata->udata.buf[0] + ntohs(ps->pr_stroffset));
		/* Make sure string is null terminated */
		*(strptr + ntohs(ps->pr_strlength)) = 0x00;

		if (statflag) {
			sprintf(statbuf, "%s", strptr);
		}

		return(pstatus);

	} /* end rcnt while */

	DBGPRT(("....printer not responding \n"))

	if (statflag) {
		sprintf(statbuf, "Printer not responding to status request\n");
	}

	return(0);
}


init_data(copts)
struct command_opt *copts;
{
	copts->input_stdin = 1;
}

/*
 * Fork a child process to gather backchannel messages while a job is active.
 */
spawn_bchanproc()
{
    int pid, ufdb;

    /*
     * Set up UDP socket.
     */
    if ((ufdb = set_udptli()) < 0) {
        printf("Failed to open UDP connection for child (status) process\n");
        exit(1);
    }

	/*
	 * Set up UDP TLI for backchannel messages.
	 */
    if (!init_backchannel(ufdb)) {
        printf("Failed to initialize backchannel process\n");
        exit(0);
    }
    if ((pid = fork()) < 0) {
        printf("Failed to spawn backchannel process\n");
        exit(1);
    }
    if (pid == 0) { /* child */
        get_bchan_msg(ufdb);
	}
    else {
        return(pid);
    }
}

/*
 * Fork a child process to gather printer status while a job is active.
 */
spawn_statproc()
{
	int pid, ufd;

	if ((pid = fork()) < 0) {
		perror("Failed to spawn status gathering process\n");
		exit(1);
	}
	if (pid == 0) { /* child */
		/*
	 	* Set up UDP socket.
	 	*/
		if ((ufd = set_udptli()) < 0) {
			printf("Failed to open UDP connection for child (status) process\n");
			exit(1);
		}

                /*
                 * if the original qef is desired call update_status otherwise
                 * use update_accounting.
                 */

                if (original)
                    update_status(ufd);
                else
                    update_accounting(ufd);

	}
	else {
		return(pid);
	}
}

/*
 * The core of the child process. Continuously queries and logs the 
 * printer status.
 */
update_status(ufd)
int ufd;
{
        int i;
        char statbuf[512], statstring[512];

        sleep(STATUSINTERVAL);

        i = 100;
        while (TRUE) {
                get_printstat(ufd, 1, statbuf);
                sprintf(statstring, "Job in progress--- Status: %s", statbuf);
                write_status(i++, statstring);
                sleep(STATUSINTERVAL);
        }
}

get_bchan_msg(ufd)
int ufd;
{
    int i, j=0, rflags, nfound, nfd;
    int pagecntr=0;
    struct t_unitdata sudata, *rudata;
    struct t_uderr *ruderr;
    char preply[MAX_REPLY_SIZE+1];
    char next_msg[MAX_REPLY_SIZE];
    struct pollfd fds[1];
    char *test1ptr, *test2ptr;
    char *cpbf;

	/*
	 * Set the time out value.
	 */
	if (copts.alternative) {
	    get_realtime();
	}

	/* 
	 * Allocate a unitdata structure and its address field.
	 */
	if ((rudata = (struct t_unitdata *)t_alloc(ufd, T_UNITDATA, T_ADDR))
       		==  (struct t_unitdata *) NULL){
       	t_error("Failed to allocate Rcv unitdata\n");
       	progexit(1);
    }

    /* Allocate a uderr structure and its buffers.
     */
    if ((ruderr = (struct t_uderr *)t_alloc(ufd, T_UDERROR, T_ALL))
          ==  (struct t_uderr *) NULL){
        t_error("Failed to allocate uderror structure\n");
        progexit(1);
    }

    /*
     * Read in the reply packet.
    */
	while (TRUE) {
		fds[0].events = fds[0].revents = 0;
		fds[0].fd = ufd;
		fds[0].events = fds[0].events | POLLIN;

		nfd = 1;

		nfound = poll(fds, nfd, POLL_TIMEOUT);

		if ((nfound < 0) || (nfound > 1)) {
			perror("Poll failed\n");
			progexit(1); 
		}
		else if (nfound == 0) {
			DBGPRT(("Timed out: Reply packet not received from printer\n"))
			continue; 
		}
		else {
			DBGPRT(("Reply packet received from printer\n"))
		}

		if (fds[0].revents & POLLIN) { 
			DBGPRT(("Poll:: Data available on UDP streams\n"))
		}

 	rudata->opt.maxlen = 0;
       	rudata->udata.maxlen = MAX_REPLY_SIZE;
        rudata->udata.buf = preply;

        if (t_rcvudata(ufd, rudata, &rflags) < 0) {
        	if (t_errno == TLOOK) {
               	if (t_rcvuderr(ufd, ruderr) < 0) {
               		 fprintf(stderr, "t_rcvuderr failed\n");
               		 progexit(1);
               	}
               	fprintf(stderr, "Bad datagram: Error=%d\n", ruderr->error);
               	progexit(1);
           	}
            else {
           		t_error("Failed to receive udata\n");
              	progexit(1);
            }
        }

        preply[MAX_REPLY_SIZE] = 0;
        if (uflag)
        if (findstr(preply, "status: ")){
            if (findstr(preply, "pagecount:"))
                if (!pagecntr){
                        cpbf = preply;
                        test1ptr=index(preply,':');
                        test1ptr++;
                        test2ptr=index(test1ptr,':');
                        test2ptr++;
                        sscanf(test2ptr,"%d",&postpagecnt);
                        pagecntr++;
                        j=0;
                        while (cpbf != test2ptr){
                                savebuf[j] = *cpbf;
                                j++;
                                cpbf++;
                        }
                        test1ptr=index(test2ptr,';');
                        sprintf(&savebuf[j]," 0%s",test1ptr);
                        j=0;
                        sprintf(preply,"%s",savebuf);
                }
                else{
                        continue;
                }
 
        }
        if (copts.alternative) {
            for (i=0;i<rudata->udata.len;i++) {
                next_msg[j++] = preply[i];
                if (preply[i] == '\n') {
                    next_msg[j] = 0;
                    if (bjobfile_present) {
                        fprintf(bjobptr, "%s%s",account.Time,next_msg);
                        fflush(bjobptr);
                    }
                    if (blogfile_present) {
                        fprintf(blogptr, "%s%s",account.Time,next_msg);
                        fflush(blogptr);
                    }
                    j = 0;
                }
            }
            if (j) {
                    next_msg[j] = 0;
                    if (bjobfile_present) {
                        fprintf(bjobptr, "%s%s\n",account.Time,next_msg);
                        fflush(bjobptr);
                    }
                    if (blogfile_present) {
                        fprintf(blogptr, "%s%s\n",account.Time,next_msg);
                        fflush(blogptr);
                    }
                    j = 0;
            }
               
        }
        else {
            if (bjobfile_present) {
                fprintf(bjobptr, "%s",preply);
                fflush(bjobptr);
            }
            if (blogfile_present) {
                fprintf(blogptr, "%s",preply);
                fflush(blogptr);
            }
        }
    }
}

char *findstr(str1,str2)
char *str1;
char *str2;
{
        char *pstr1;
        char *pstr2;
 
        for( ;*str1;str1++) {
                pstr1 = str1;
                pstr2 = str2;
                for( ;(*pstr1 == *pstr2) && *pstr2;++pstr1,++pstr2);
                if(*pstr2 == '\0')
                        return(str1);
        }
        return(0);
}

get_realtime()
{
 
        int j, k, tmp;
        long clock;
        static char month[12][4]={"Jan","Feb","Mar","Apr",
                                  "May","Jun","Jul","Aug",
                                  "Sep","Oct","Nov","Dec"};
        struct time_info alttime;
        char temptime[7];
 
        /*
         * Save the time.
         */
        clock = time((long *)0);
        strcpy(account.Time, ctime(&clock));
        tmp = strlen(account.Time);
        account.Time[tmp-1] = 0;
        if (copts.alternative) {
            sscanf(account.Time,"%s %s %s %s %s",
                   alttime.day,
                   alttime.month,
                   alttime.daynum,
                   alttime.hrminsec,
                   alttime.year);
 
            /* Get Month */
            for (j=1;j<14;j++) {
                   if (!strcmp(month[j-1],alttime.month)) {
                       if ((j-1)<10)
                           sprintf(account.Time,"0%d",j);
                       else sprintf(account.Time,"%d",j);
                       break;
                   }
            }
 
            /* Get Day of Month */
            if (j=(atoi(alttime.daynum)) < 10) {
                sprintf(temptime,"0%s",alttime.daynum);
                strcat(account.Time,temptime);
            }
            else {
               sprintf(temptime,"%s",alttime.daynum);
               strcat(account.Time,temptime);
            }
 
            /* Get Hours, Minutes, and Seconds */
            k=0;
            for (j=0;j<strlen(alttime.hrminsec);j++) {
               if (alttime.hrminsec[j] != ':')
                   temptime[k++] = alttime.hrminsec[j];
            }
            temptime[6] = 0;
            strcat(account.Time,temptime);
        }
 
}

update_accounting(ufd)
int ufd;
{
	int i, items_got=0, pos=0, tp, pstatus, length;
    long clock;
	char statstring[512];
    char old_statstring[512];
    char temp[160], temparray[512];
    int indx_ptr, old_page, old_Recvd = 1;
	char old_status[MODNAME_LEN];
	char statbuf[512];
	char finalstat[525];
	int count = 0;
	char *temp_ptr;
	int no_error_status = 0;

	sleep(copts.granularity);
        old_page = 0;
	i = 100;
        while (TRUE) {
                get_printstat(ufd, 1, statbuf, 0);
                strcpy(statstring, statbuf);
		temp_ptr = statstring;
 
                /*
                 * Get the User and Title info, if available
                 *
                 */
                for (i=0;temp_ptr[i];i++)
                    if (temp_ptr[i] == '\n')
                    {
                        no_error_status = 1;
                        break;
                    }
 
                if (no_error_status)
                {
                        while (count < 7) {
                                while (*temp_ptr++ != '\n');
                                ++count;
                        }
                        if (*temp_ptr == '\n')
                        {
                                strcpy(temp,++temp_ptr);
                        }
                        else strcpy(temp, temp_ptr);
                        count = 0;
                        no_error_status = 0;
                }

                /*
                 * Scan the remainder of statstring into
                 * the appropriate field of the structure.
                 */
                items_got = sscanf(statstring,"%d %d %d %d %d %d %d",
                                   &accounting.Status,
                                   &accounting.Page,
                                   &accounting.Recvd,
                                   &accounting.Sheet,
                                   &accounting.JobId,
                                   &accounting.ByteSize,
                                   &accounting.Emulation);
                 /*
                  * Process the user and job title info.  The temp array will contain
                  * either the Title or the User name (or both), separated by a <LF>
                  */
 
                 if (strlen(temp) > 2){
                      indx_ptr = 0;
                      pos = 0;
                      while( temp[indx_ptr] != '\n')
                          temparray[pos++] = temp[indx_ptr++];
                      temparray[pos] = 0;
                      pos=0;
                      indx_ptr++;
                      if (temp[indx_ptr])       /* if there is no username, we will be at the end of statstring */
                                                /* after having just read in the title                          */
                      {
                        strcpy(accounting.For, temparray);
                        while(( temp[indx_ptr] != '\n')&&(indx_ptr <= strlen(temp)))
                                accounting.Title[pos++] = temp[indx_ptr++];
 
                        if (pos == 0) /* we have a username only */
                                sprintf(accounting.Title,"Untitled");
                        else
                                accounting.Title[pos] = 0;
                      }

                      else /* we have a title only */
                      {
                        sprintf(accounting.For, "Unknown");
                        strcpy(accounting.Title, temparray);
                      }
                      length = strlen(temp);
                      for (i=0;i<length;i++)
                           temp[i]=0;
                 }
                 else{ /* no username or title */
                     sprintf(accounting.For,"Unknown");
                     sprintf(accounting.Title,"Untitled");
                 }

                 /*
                  * If real time is desired the copy in the
                  * System time of the message.
                  */

                 if (copts.time)
		    get_realtime();

                 /*
                  * Test the accounting.Status bit for printer status.
                  * Then copy equivalent string into account.Status.
                  */

                 if (items_got > 2){
                     if (accounting.Status & 0x20)
                        strcpy(account.Status,"waiting");
                     else
                     if (accounting.Status & 0x10)
                     {
                        if (copts.alternative)
                                strcpy(account.Status,"idle");
                        else
                                strcpy(account.Status,"terminating");
                     }
                     else
                     if (accounting.Status & 0x8)
                     {
                        if (copts.alternative)
                                strcpy(account.Status,"processing");
                        else
                                strcpy(account.Status,"printing");
                     }
                     else
                     if (accounting.Status & 0x4)
                        strcpy(account.Status,"rasterizing");
                     else
                     if (accounting.Status & 0x2)
                     {
                        if (copts.alternative)
                                strcpy(account.Status,"busy");
                        else
                                strcpy(account.Status,"interpreting");
                     }
                     else
                     if (accounting.Status & 0x1)
                        strcpy(account.Status,"spooling");

                     if (accounting.Recvd & 0x10 )
                        strcpy(account.Recvd,"Optional I/O");
                     else
                     if (accounting.Recvd == 13 )
                        strcpy(account.Recvd,"Parallel");
                     else
                     if (accounting.Recvd == 12 )
                        strcpy(account.Recvd,"Serial");
                     else
                        strcpy(account.Recvd,"Unknown");
                     /*
                      * Create the output string.
                      * Or copy in the error message.
                      */
                     if (copts.alternative)
                        sprintf(statstring, "%s%%%%[status:%s; source:Ethernet:%s; pagecount:%d; user:%s; title:%s]%%%%",
                                account.Time,
                                account.Status,
                                copts.hostname,
                                accounting.Sheet,
                                accounting.For,
                                accounting.Title);
                     else
                        sprintf(statstring, "%%%%[status:%s; pagecount:%d; interface:%s:%s; user:%s; title:%s; time:%s]%%%%",
                                account.Status,
                                accounting.Sheet,
                                account.Recvd,
                                copts.hostname,
                                accounting.For,
                                accounting.Title,
                                account.Time);
                 }
                 else {
                      if (copts.alternative) {
                          if (findstr(statstring,"FEED JAM")||findstr(statstring,"INPUT BIN JAM"))
                              sprintf(statstring, "%%%%[PrinterError:paper entry misfeed]%%%%");
                          else if (findstr(statstring,"JAM"))
                                   sprintf(statstring, "%%%%[PrinterError:paper jam]%%%%");
                               else if (findstr(statstring,"ADJUST INPUTBIN")||findstr(statstring,"ADJUST INPUT BIN"))
                                        sprintf(statstring, "%%%%[PrinterError:no paper tray]%%%%");
                                    else if(findstr(statstring,"LOW TONER"))
                                             sprintf(statstring, "%%%%[PrinterError:toner low]%%%%");
                                         else if(findstr(statstring,"TONER OUT"))
                                                  sprintf(statstring, "%%%%[PrinterError:out of toner]%%%%");
                                              else if(findstr(statstring,"PUT")&& findstr(statstring,"PAPER"))
                                                       sprintf(statstring, "%%%%[PrinterError:out of paper]%%%%");
                                                   else sprintf(statstring, "%%%%[PrinterError:%s]%%%%",statbuf);
                      }
                      else sprintf(statstring, "%%%%[PrinterError:%s]%%%%", statbuf);
                      if (strcmp(old_statstring, statstring) && !findstr(statstring,"IDLE") &&
				 !findstr(statstring,"NO ERROR")){
                      if (copts.alternative) {
                         sprintf(finalstat,"%s%s",account.Time,statstring);
                         write_accounting(1,finalstat);
                      }
                      else
                         write_accounting(i++, statstring);
                      }
                      if (accounting.Terminate == 1)
                          exit(0);
                 }

                 /*
                  * We don't want to repeat old info... so only report
                  * info if the page changed or the status changed.  Also
                  * only report status pertaining to opt I/O.
                  */
                 if (verbose)
                     if ((accounting.Recvd & 0x10)&&
                        ((accounting.Page != old_page)||
			 (strcmp (account.Status,old_status) != 0) ||
			 (old_Recvd == 0)))
                             write_accounting(i++, statstring);

                /*
                 * If the terminating message has been received then
                 * write the last status message (non-verbose) and exit
                 */

                if ((items_got >= 1)&&(accounting.Recvd & 0x10))
                    if (accounting.Status & 0x10){
                        if (!verbose)
                            write_accounting(i++, statstring);
                        pstatus = get_printstat(ufd, 1, statbuf, 1);
                /*
                 * If modified backchannel message needs to be sent to
                 * a file then do it.
                 */
                if (uflag)
                if (copts.alternative){
                        get_realtime();
                    if (bjobfile_present) {
                        fprintf(bjobptr, "%s%%%%[status: finished job; pagecount: %d;]%%%%\n",account.Time,accounting.Sheet);
                        fflush(bjobptr);
                    }
                    if (blogfile_present) {
                        fprintf(blogptr, "%s%%%%[status: finished job; pagecount: %d;]%%%%\n",account.Time,accounting.Sheet);
                        fflush(blogptr);
                    }
                }
                else
                {
                    if (bjobfile_present) {
                        fprintf(bjobptr, "%%%%[status: finished job; pagecount: %d;]%%%%\n",accounting.Sheet);
                        fflush(bjobptr);
                    }
                    if (blogfile_present) {
                        fprintf(blogptr, "%%%%[status: finished job; pagecount: %d;]%%%%\n",accounting.Sheet);
                        fflush(blogptr);
                    }
                }
                        exit(0);
                    }
                if (accounting.Terminate == 1)
                      exit(0);
                
                old_page = accounting.Page;
		old_Recvd = accounting.Recvd;
		strcpy (old_status,account.Status);
                sprintf(old_statstring, "%s",statstring);
                sleep(copts.granularity);

	}
}

/*
 * Terminate the child process.
 */
kill_statproc()
{
	if (kill(copts.cpid[0], SIGKILL) < 0) {
		if (errno != ESRCH) {
			perror("Failed to terminate status gathering process\n");
		exit(1);
		}
	}
	while (wait(0) != copts.cpid[0]);
}

kill_bchanproc()
{
    if (kill(copts.cpid[1], SIGKILL) < 0) {
        if (errno != ESRCH) {
            perror("Failed to terminate the backchannel process(3)\n");
            exit(1);
        }
    }
}

/*
 * Write printer status to the status file.
 */
write_accounting(pstat, msg)
int pstat;
char *msg;
{
	long clock;
	FILE *fptr;

	DBGPRT(("Status Message:: %s\n", msg))

	if (( fptr = fopen(copts.statfile, "w"))  ==  (FILE *)NULL) {
		perror("Failed to open status file\n");
		progexit(1);
	}

	clock = time((long *)0);

	fprintf(fptr, "%s  %s\n", ctime(&clock), msg);
	fflush(fptr);
        if (jobfile_present) {
            fprintf(jobptr, "%s\n", msg);
            fflush(jobptr);
        }

        if (logfile_present) {
            fprintf(logptr, "%s\n", msg);
            fflush(logptr);
        }

	if (fclose(fptr) < 0) {
		perror("Failed to close status file\n");
		progexit(1);
	}
}


write_status(pstat, msg)
int pstat;
char *msg;
{
        long clock;
        FILE *fptr;
        DBGPRT(("Status Message:: %s\n", msg))

        if (( fptr = fopen(copts.statfile, "w"))  ==  (FILE *)NULL) {
                perror("Failed to open status file\n");
                progexit(1);
        }

        clock = time((long *)0);

        fprintf(fptr, "%s  %s\n", ctime(&clock), msg);
        fflush(fptr);

        if (fclose(fptr) < 0) {
                perror("Failed to close status file\n");
                progexit(1);
        }
}

process_message(message, old_stat)
char *message;
char *old_stat;
{
int items_got;
char statstring[512];
char finalstat[525];

                        sprintf(statstring, "%s", message);

                /*
                 * Scan the string returned from the printer into
                 * the appropriate field of the structure.
                 */

                items_got = sscanf(statstring,"%d %d %d %d %d %d %d",
                                   &accounting.Status,
                                   &accounting.Page,
                                   &accounting.Recvd,
                                   &accounting.Page,
                                   &accounting.JobId,
                                   &accounting.ByteSize,
                                   &accounting.Emulation
                                   );
                      if (items_got < 2){
                          if (copts.alternative) {
                              if (findstr(message,"FEED JAM")||findstr(statstring,"INPUT BIN JAM"))
                                  sprintf(statstring, "%%%%[PrinterError:paper entry misfeed]%%%%");
                              else if (findstr(message,"JAM"))
                                       sprintf(statstring, "%%%%[PrinterError:paper jam]%%%%");
                                   else if (findstr(message,"ADJUST INPUTBIN")||findstr(statstring,"ADJUST INPUT BIN"))
                                            sprintf(statstring, "%%%%[PrinterError:no paper tray]%%%%");
                                        else if(findstr(message,"LOW TONER"))
                                                 sprintf(statstring, "%%%%[PrinterError:toner low]%%%%");
                                             else if(findstr(message,"TONER OUT"))
                                                      sprintf(statstring, "%%%%[PrinterError:out of toner]%%%%");
                                                  else if(findstr(message,"PUT")&& findstr(statstring,"PAPER"))
                                                           sprintf(statstring, "%%%%[PrinterError:out of paper]%%%%");
                                                       else sprintf(statstring, "%%%%[PrinterError:%s]%%%%",message);
                          }
                          else sprintf(statstring, "%%%%[PrinterError:%s]%%%%", message);
                          if (strcmp(old_stat, statstring) && !findstr(statstring,"IDLE") &&
				     !findstr(statstring,"NO ERROR")){
                              sprintf(old_stat,"%s",statstring);
                              if (copts.alternative) {
                                 get_realtime();
                                 sprintf(finalstat,"%s%s",account.Time,statstring);
                                 write_accounting(1,finalstat);
                              }
                              else
                                 write_accounting(1, statstring);
                          }
                      }
}

progexit(exitcode)
int exitcode;
{
	if (copts.cpid[0] != getpid()) { /*Not child process -- parent process. */
		kill_statproc();
	}
	if (copts.cpid[1] != getpid()) {
		kill_bchanproc();
	}
	exit(exitcode);

}

