/***********************************************************************\
* Title: prtpost							*
* Date: 15-September-1988						*
* Author: H.V. Stentz							*
* Description:								*
*	This is a program to read a raw text file and format it	for a	*
*	PostScript printer. It takes an optional file name as an argu-	*
*	ment and writes to standard output. If no input file is given,	*
*	standard input is assumed.					*
*									*
* Revisions:								*
*	15-September-1988	--Initial release--			*
\***********************************************************************/
#include <stdio.h>
#include <time.h>
#define TRUE 1
#define FALSE 0
#define FOREVER for(;;)
#define GSAVE printf("gsave ")
#define TRANSLATE(x,y) printf("%d %d translate ", x, y)
struct page_dim {
	int w;	/* width */
	int l;	/* length */
};
	
/* global variables */
FILE *infile;			/* input file pointer */
FILE *eoppnt;			/* end-of-page file pointer */
FILE *boppnt;			/* beginning-of-page file pointer */
char fname[40] = "";		/* input file name */
char eopfile[40] = "";		/* end-of-page file name */
char bopfile[40] = "";		/* beginning-of-page file name */
short overlay = FALSE;		/* flags end-of-page file is present */
short underlay = FALSE;		/* flags beginning-of-page file is present */
short twocol = FALSE;		/* set text in two columns */
short fourup = FALSE;		/* print four half-size pages per sheet */
short pass = 0;			/* pass counter for 2-column mode */
short rule = TRUE;		/* separate columns with a rule line */
short frule = TRUE;		/* separate 4-up pages with rule lines */
short numline = FALSE;		/* prefix lines with line numbers */
unsigned long linenum = 0;	/* line number */
int copies = 1;			/* number of copies */
char date_time[40];		/* time and date */
int pageno = 1;			/* page number */
int page_w = 6120;		/* page width (decipoints) */
int page_l = 7920;		/* page length (decipoints) */
int paper_w = 6120;		/* paper width -- independent of orientation */
int paper_l = 7920;		/* paper length */
int tab = 8;			/* tab stop */
char font[40] = "Courier";	/* font */
int ps = -1;			/* point size (decipoints) */
int lspace = -1;		/* line spacing (decipoints): defaults
				   to point size */
/* page sizes:
	0	letter (8.5" X 11")
	1	legal (8.5" X 14")
	2	ledger (11" X 17")
	3	A3 (297 mm X 420 mm)
	4	A4 (210 mm X 297 mm)
	5	B5 (182 mm X 257 mm)
*/
int page_size = 0;
/* table of page widths and lengths. index matches page codes given above. */
struct page_dim page_aray[] = {
	{6120, 7920}, {6120, 10080}, {7920, 12240},
	{8418, 11905}, {5952, 8418}, {5159, 7285}
};
int squeeze = 100;		/* character compression, expressed as %
				   of normal width */
short header = TRUE;		/* switch to control printing of header */
char hfont[40] = "Courier-Bold";	/* header font */
int hps = 100;			/* header point size */
int hsqueeze = 100;		/* header character compression */

/* All positions expressed as decipoints. Horizontal positions measured from
   left edge of page, vertical positions measured up from bottom of page.
   variables initialized to -1 depend on the page_size, which may be
   changed from its default of 0 (letter) by a command line option.
 */
short landscape = FALSE;	/* switch for landscape mode */
int lm = -1;			/* body left margin */
int tm = -1;			/* body top margin */
int bm = 360;			/* body bottom margin */
int hlm = 360;			/* header left margin */
int hrm = -1;			/* header right margin */
int htm = -1;			/* header top margin */
int vpos;			/* current vertical position */

main(argc, argv)
int argc;
char *argv[];
{
	FILE *fopen();
	char *strcpy();
	int i;

	getargs(argc, argv);	/* get options from command line */
	validate();		/* validate parameters and initialize */

	/* open input file */
	if(fname[0] != '\0')
	{
		if (!(infile = fopen(fname, "r")))
		{
			fprintf(stderr, "Couldn't open input file %s\n", fname);
			exit(1);
		}
	}
	else
	{
		/* assume standard input if no file name given */
		infile = stdin;
		strcpy(fname, "standard input");
	}

	/* open beginning-of-page file */
	if(bopfile[0] != '\0')
	{
		if (!(boppnt = fopen(bopfile, "r")))
		{
			fprintf(stderr,
			 "Couldn't open beginning-of-page file %s\n", bopfile);
			exit(1);
		}
		underlay = TRUE;
	}

	/* open end-of-page file */
	if(eopfile[0] != '\0')
	{
		if (!(eoppnt = fopen(eopfile, "r")))
		{
			fprintf(stderr,
			 "Couldn't open end-of-page file %s\n", eopfile);
			exit(1);
		}
		overlay = TRUE;
	}

	/* send the prologue */
	prologue();

	/* main loop */
	while (!feof(infile))
	{
		/* save VM state, scale units to decipoints, set line width */
		printf("/sv save def .1 .1 scale 4.8 setlinewidth\n");

		if(fourup)	/* check for four-up option */
		{
			/* if frule is set, draw a pair of lines in a cross
			 to separate the pages */
			if(frule)
				printf("%d 0 m %d %d l 0 %d m %d %d l stroke\n",
					paper_w/2, paper_w/2, paper_l,
					paper_l/2, paper_w, paper_l/2);

			/* Shrink the coordinate system slightly, so that a
			 * border 1/4" (180 d.p.) on all sides is left clear.
			 * This prevents printing in the unimageable area.
			 */
			printf("180 180 translate\n");
			printf("%d 360 sub %d div %d 360 sub %d div scale\n",
				paper_w, paper_w, paper_l, paper_l);

			/* Locate and scale each page with placepage(), then
			 * print it with dopage()
			 */
			for(i=1; i<=4; i++)
			{
				placepage(i);
				dopage();
				printf("grestore\n");
				if(feof(infile))
					break;
			}
		}
		else
			/* just do one page */
			dopage();

		/* showpage and restore old VM state */
		printf("sp sv restore\n");
	}
	printf("\004");		/* end of job */
}

prologue()	/* send out the prologue */
{
	printf("%%!\n");	/* tells spooler to execute PostScript file */
	/* Scale the main font, compressing if reqd. Font dict stored to F */
	printf("/F /%s findfont [%d %d mul 100 div 0 0 %d 0 0] makefont def\n",
		font, ps, squeeze, ps);
	/* Same for header font; font dict stored to hF */
	printf("/hF /%s findfont [%d %d mul 100 div 0 0 %d 0 0] makefont def\n",
		hfont, hps, hsqueeze, hps);
	/* define shorthand names for common operators */
	printf("/s /show load def /sp /showpage load def /m /moveto load def\n");
	printf("/l /lineto load def /cp /closepath load def /np /newpath load def\n");
	/* cs = center show: show a string centered around the given x-coord */
	printf("/cs {dup stringwidth pop 2 div neg 3 -1 roll add\n");
	printf("currentpoint exch pop m s} def\n");
	/* rs = right show: right edge of string placed at given x-coord */
	printf("/rs {dup stringwidth pop neg 3 -1 roll add\n");
	printf("currentpoint exch pop m s} def\n");
	/* define DT (date and time) and FL (filename) strings */
	printf("/DT (%s) def\n/FL (%s) def\n", date_time, fname);
	/* set number of copies */
	if(copies > 1)
		printf("/#copies %d def\n", copies);
}

start_page()	/* start of page functions */
{
	char line[134];

	/* two-column output is done in two passes */
	if (pass == 0)
	{
		if(landscape)	/* rotate coordinate system */
			printf("90 rotate 0 -%d translate\n", page_l);
		printf("/PG (%d) def\n", pageno);
		if(underlay)	/* send beginning-of-page file */
		{
			FOREVER
			{
				if(!fgets(line, sizeof(line), boppnt))
					break;
				fputs(line, stdout);
			}
			rewind(boppnt);
		}
		if (header)	/* format standard page header */
		{
			printf("hF setfont %d %d m DT s\n", hlm, htm);
			printf("%d FL cs\n", (hlm + hrm)/2);
			printf("%d (Page %d) rs\n", hrm, pageno++);
		}
		printf("F setfont\n");	/* set main font */
	}
	if(twocol)	/* special action for two-column mode */
	{
		if(pass == 0)
		{
			/* clip output to left half of page */
			GSAVE;
			clip(0, 0, page_w/2, page_l);
		}
		else
			/* shift coordinate system to right half of page */
			printf("grestore gsave %d 0 translate\n", page_w/2);
	}
	else
		pass++;	/* increment pass counter to force only one pass */
	vpos = tm;	/* set vertical position to top margin */
}


getline(s, lim) /* Read in a line from infile, checking for */
char *s;	/* FF anywhere in the line.  If FF is found,*/
int lim;	/* send it back (by itself) to lineout.     */
{
	int c, i;
	
	for (i=0; i<lim-1 && (c=getc(infile)) != EOF && c != '\n' 
			  && c != '\014'; ++i)
	    s[i] = c;		

	if (c == '\n')
		s[i] = c;

	if (c == '\014') 
		if (i != 0) /* FF found in mid-line or at end of line*/
		{
		        s[i] = '\n';
			ungetc(c, infile);
	 	}
		else /* FF found at beginning of line */
		{
			s[i] = c;
			++i;
			s[i] = '\n';
		}
	++i;
	s[i] = '\0';	
}

lineout()	/* read and print a line */
{
	char line[200];
	getline (line, sizeof(line));	/* read in a line from infile */
	line[strlen(line) - 1] = '\0';	/* get rid of newline */
	if(line[0] == '\014')		/* check for form feed */
		vpos = 0;		/* force page break */
	else if(line[0] != '\0' || numline)	/* suppress blank lines */
	{
		expand(line);		/* expand tabs */
		printf("%d %d m (", lm, vpos);	/* position line */
		if (numline)	/* number lines? */
			printf("%05d ", ++linenum);
		printf("%s) s\n", line);
	}
	vpos = vpos - lspace;	/* move down by one line space */
}

expand(line)	/* expand tab stops */
char *line;
{
	char *strcpy();
	char expline[250];
	char *cpt;
	int column;
	int spcnum, i;
	char c;

	column = 1;
	cpt = expline;
	strcpy(expline, line);	/* make local line copy */
	while((c = *cpt++) != '\0')
	{
		if (c != '\t')	/* not tab char */
		{
			/* prefix "(", ")", and "\" with "\" to prevent
			 * syntax errors
			 */
			if (c == '(' || c == ')' || c == '\\')
				*line++ = '\\';
			*line++ = c;
			column++;	/* inc column count */
		}
		else	/* tab character found */
		{
			/* round column to next highest multiple of tab, then
			 * find spcnum = number of spaces needed to get there
			 */
			spcnum = ((column + tab - 1)/tab)*tab - column + 1;
			for(i=0; i<spcnum; i++)	/* stick in the spaces */
				*line++ = ' ';
			column += spcnum;	/* update column position */
		}
	}
	*line = '\0';	/* terminate the output string */
}

end_page()	/* end of page functions */
{
	char line[134];

	if(pass > 0)	/* second pass? */
	{
		if(twocol)
		{
			printf("grestore\n");	/* restore initial coords */

			/* draw rule line down center of page */
			if(rule)
			 printf("%d %d m %d %d l stroke\n",
			   page_w/2, tm + ps, page_w/2, bm);
		}

		if(overlay)	/* send end-of-page file */
		{
			FOREVER
			{
				if(!fgets(line, sizeof(line), eoppnt))
					break;
				fputs(line, stdout);
			}
			rewind(eoppnt);
		}
	}
}

/* Validate parameters and initialize. Replace negative parameters
 * with default values.
 */
validate()
{
	long time_secs;
	char *strcpy();

	if(tab < 1)
		err("tab stop must be at least 1\n");
	if(ps < 0)
	{
		if(twocol && landscape)
			ps = 70;	/* two col. landscape default 7 pt. */
		else
			ps = 100;	/* normal default is 10 pt. */
	}
	else if(ps < 30)
		err("body point size must be at least 30\n");
	if(hps < 30)
		err("header point size must be at least 30\n");
	if(lspace < 0)
		lspace = ps;
	else if(lspace < 1)
		err("lines per inch (hundredths) must be at least 1\n");
	else
		/* convert line per inch to decipoints */
		lspace = 72000L / (long)lspace;
	if(squeeze < 1 || squeeze > 500)
		err("body font compression must be between 1% and 500%\n");
	if(hsqueeze < 1 || hsqueeze > 500)
		err("header font compression must be between 1% and 500%\n");
	if(page_size < 0 || page_size > 5)
		err("page size must be between 0 and 5\n");

	/* get page dimensions and margins */
	paper_w = page_aray[page_size].w;
	paper_l = page_aray[page_size].l;
	if(landscape)
	{
		page_w = paper_l;
		page_l = paper_w;
	}
	else
	{
		page_w = paper_w;
		page_l = paper_l;
	}
	if(lm < 0)
	{
		if(twocol)
			lm = 270;
		else
			lm = 540;
	}
	if(tm < 0)
		tm = page_l - 720;
	if(htm < 0)
		htm = page_l - 360;
	if(hrm < 0)
		hrm = page_w - 360;
	if(tm <= bm)
		err("inconsistent top and bottom margins\n");
	if(htm < tm)
		err("header below main body of text\n");
	if(hlm >= hrm)
		err("inconsistent header left and right margins\n");
	/* get current date and time */
	time(&time_secs);
	strcpy(date_time, ctime(&time_secs));
	date_time[strlen(date_time) - 1] = '\0';	/* strip newline */
}

/* bad parameter error */
err(estring)
char *estring;
{
	fprintf(stderr, "Bad parameter: %s", estring);
	exit(1);
}

clip(llx, lly, urx, ury)	/* establish clipping path */
int llx;	/* lower left coordinates */
int lly;
int urx;	/* upper right coordinates */
int ury;
{
	printf("%d %d m %d %d l %d %d l %d %d l cp clip np\n",
		llx, lly, urx, lly, urx, ury, llx, ury);
}

placepage(count)	/* set up coordinates for 1/2 size pages */
int count;
{
	/* order of pages: 1 = top left, 2 = top right, 3 = bottom left,
	 * 4 = bottom right. Note that the translation and clipping are
	 * done BEFORE the coordinate system is rotated to landscape mode,
	 * so the pages are placed differently for landscape and portrait.
	 */
	GSAVE;
	switch(count)	/* switch on page number */
	{
		case 1:
			if(!landscape)
				TRANSLATE(0, paper_l/2);
			break;
		case 2:
			if(landscape)
				TRANSLATE(0, paper_l/2);
			else
				TRANSLATE(paper_w/2, paper_l/2);
			break;
		case 3:
			if(landscape)
				TRANSLATE(paper_w/2, 0);
			break;
		case 4:
			if(landscape)
				TRANSLATE(paper_w/2, paper_l/2);
			else
				TRANSLATE(paper_w/2, 0);
			break;
	}
	clip(0, 0, paper_w/2, paper_l/2);
	printf(".5 .5 scale\n");
}

dopage()	/* format logical page */
{
	for(pass=0; pass<2; pass++)
	{
		start_page();	/* do start of page functions */
		while(!feof(infile))
		{
			lineout();	/* read and print a line */
			if (vpos < bm)	/* check for page bottom */
				break;
		}
		if(feof(infile))
			pass++;
		end_page();	/* do end of page functions */
	}
}
