/* Copyright (c) 2003 KONICA MINOLTA BUSINESS TECHNOLOGIES, INC. All rights reserved. */
/* 
 * BSD socket implementation of ies.
 * The program reads either its stdin or the file specified on the command
 * line and prints the file. Before sending the data, printer status is checked.
 * To be compatible with the server side, a connection is opened for each file
 * to be transferred. 
 */


#ifdef SOLARIS
#include <sys/fcntl.h>
#endif
#include <stdio.h>
#include <netdb.h>
#include <sys/file.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/errno.h>
#include <netinet/in.h>
#include <signal.h>
#include "qef.h"

#ifndef FD_ZERO
#include "selmacro.h"
#endif

char *index();

struct sockaddr_in printaddr;
struct command_opt copts;
char savebuf[MAX_REPLY_SIZE+1];
int postpagecnt=0;
extern int errno;
extern short debug, 
             nostat,
             bjobfile_present,
             blogfile_present,
             jobfile_present,
             logfile_present,
             verbose,
	     uflag,
             original;

FILE *bjobptr;
FILE *blogptr;
FILE *jobptr;
FILE *logptr;
struct ACCOUNT   account;
struct ACCOUNTING accounting;


main(argc, argv)
int argc;
char *argv[];
{
	int tpfd, ufdm, cpid, pstatus, sleeptime;
	int child_status;
        char old_statstring[512], statbuf[512];
	struct sockaddr_in  claddr;
	struct hostent    *printer;
	struct file_list *flist;

	init_data(&copts);
	proc_args(argc, argv, &copts); 

        write_status(STAJSTART, "Starting job");
        if (!original) {
            /*
             * If a log file was specified on the command line then
             * open the file in order to append the status messages returned
             */
            if (logfile_present) {
                if (( logptr = fopen(copts.logfile, "a"))  ==  (FILE *)NULL) {
                        perror("Failed to open log file\n");
                        progexit(1);
                 }
            }

            /*
             * If a job file was specified on the command line then 
             * open the file in order to write the status messages to it.
             */

            if (jobfile_present) {
                if (( jobptr = fopen(copts.jobfile, "w"))  ==  (FILE *)NULL) {
                        perror("Failed to open job file\n");
                        progexit(1);
                 }
            }
        }

        /*
         * If a backchannel job file was specified on the command line then 
         * open the file in order to write the backchannel messages to it.
         */

        if (bjobfile_present && (strcmp(copts.bjobfile,copts.jobfile) != 0)) {
            if (( bjobptr = fopen(copts.bjobfile, "w"))  ==  (FILE *)NULL) {
                perror("Failed to open backchannel job file\n");
                progexit(1);
             }
        }
        else {
            if (bjobfile_present) {
                bjobptr = jobptr;
            }
        }

        /*
         * If a backchannel log file was specified on the command line then 
         * open the file in order to add the backchannel messages to it.
         */

        if (blogfile_present && (strcmp(copts.blogfile,copts.logfile) != 0)) {
            if (( blogptr = fopen(copts.blogfile, "a"))  ==  (FILE *)NULL) {
                perror("Failed to open backchannel log file\n");
                progexit(1);
            }
        }
        else {
            if (blogfile_present) {
                blogptr = logptr;
            }
        }

	if ((printer = gethostbyname(copts.hostname)) == (struct hostent *)NULL) {
		fprintf(stderr, "gethostbyname failed:hostname not found in /etc/hosts\n");
		exit(1);
	}

	DBGPRT(("printer hostent fields:\n h_name= %s h_addr= %s h_length=0x%x\n", 
		printer->h_name, inet_ntoa((struct in_addr *)(printer->h_addr)), printer->h_length))

	/*
	 * Set up the address of the peer process with whom we want to
	 * communicate. The same address (printaddr) can be used for both
	 * TCP and UDP connections since both tcp port and udp port numbers
	 * are the same.
	 */

	jzero((char *)&printaddr, sizeof(printaddr)); 
	printaddr.sin_family = AF_INET;
	printaddr.sin_port = htons(copts.port_number);
 	jcopy((char *)printer->h_addr, (char *)&printaddr.sin_addr, printer->h_length);

	
	flist = copts.flist;
	while (copts.input_stdin || (flist != (struct file_list *)NULL)) {

        /*
         * Set up UDP socket.
         */
        if ((ufdm = setup_udp()) < 0) {
            printf("Failed to open UDP connection\n");
            progexit(1);
        }

        DBGPRT(("UDP file descriptor = %d\n", ufdm))


        /*
         * Get printer status using UDP based STATUS1 protocol.
         */
        pstatus = get_printstat(ufdm, 1, statbuf, 0);
        sleeptime = SLEEPTIME;

        while (pstatus == 0) {
            sleep(sleeptime);
            pstatus = get_printstat(ufdm, 1, statbuf, 0);
            if (!original)
                process_message(statbuf, old_statstring);
        }

        /*
         * Initialize the backchannel socket and exchange the protocol
         * packet and let the DB know if we want backchannel for this job.
         */
        ufdm = setup_backchannel(); 
        if (!init_backchannel(ufdm)) {
		   exit(0);
        }

        /*
         * Spawn the child process for obtaining status1 info
         * from the printer.
         */
        if (!nostat) {
            copts.cpid[0] = spawn_statproc();
        }

        /*
         * Spawn the child process for backchannel messages from the printer.
         */
        if (bjobfile_present || blogfile_present) {
            DBGPRT(("Spawning the backchannel process\r\n"))
            copts.cpid[1] = spawn_bchanproc(ufdm);
        }
        else {
		    DBGPRT(("Backchannel process not started, wrong DB version\n"))
        }

        /*
         * Open a tcp socket.
         */
        if ((tpfd = socket(PF_INET, SOCK_STREAM, 0)) < 0) {
            perror("Failed to open a socket\n");
            progexit(1);
        }
	
        DBGPRT(("TCP file descriptor = %d\n", tpfd))

        /*
         * Connect (TCP) to the peer on the printer.
         */
        if (connect(tpfd, &printaddr, sizeof(printaddr)) < 0) {
		    perror("Failed to connect to printer\n");
		    progexit(1);
        }

        /*
         * Printer is accepting data; Transfer it (over TCP).
         */
        if (xfer_data(tpfd, &copts, flist) < 0) {
            printf("Data Transfer to the printer failed\n");
            write_status(STAACOMP, "communications ended abnormally");
            progexit(1);
        }

        if (close(tpfd) < 0) {
		    perror("Failed to close tcp socket \n");
		    progexit(1);
        }
	
        if (close(ufdm) < 0) {
            perror("Failed to close udp socket \n");
            progexit(1);
        }

        /*
         * Wait for the child to receive a terminating status
         * request from the printer.
         */
        if (!original) {
#ifndef SGI
	        while (wait(0) != copts.cpid[0]);
#endif
#ifdef SGI
            wait(&child_status);
#endif
        }

        write_status(STAJCOMP, "Job transmission completed");

    if (original) {
	/*
         * Terminate the status gathering and backchannel processes
	 */
        if (!nostat) {
            kill_statproc();
        }
    }

    /* Terminate the backchannel process */
    if (bjobfile_present || blogfile_present) {
        if (kill(copts.cpid[1], SIGKILL) < 0) {
            if (errno != ESRCH) {
                perror("Failed to terminate the backchannel process(2)\n");
                exit(1);
            }
        }
    }

		/*
         * Disable stdin so that next loop fails.
		 */
        if (copts.input_stdin) {
            copts.input_stdin = 0;
        }
        else {
            flist = flist->next;
        }
		
    } /* end while file list */

    if (!original) {

        /*
         * If we are using accounting, close any open status files.
         */
        if (jobfile_present) {
            if (fclose(jobptr) < 0) {
                perror("Failed to close status file\n");
                progexit(1);
            }
        }
        if (logfile_present) {
            if (fclose(logptr) < 0) {
                perror("Failed to close status file\n");
                progexit(1);
            }
        }
    } 

    /*
     * If we are using backchannel, close any open message files.
     */
    if (bjobfile_present && (strcmp(copts.bjobfile,copts.jobfile) != 0)) {
        if (fclose(bjobptr) < 0) {
            perror("Failed to close backchannel job file\n");
            progexit(1);
        }
    }
    if (blogfile_present && (strcmp(copts.blogfile,copts.logfile) != 0)) {
        if (fclose(blogptr) < 0) {
            perror("Failed to close backchannel log file\n");
            progexit(1);
        }
    }

    write_status(STASTERM, "Spooler terminated communication");
	exit(0);
}

xfer_data(tpfd, copts, flist)
int tpfd;
struct command_opt *copts;
struct file_list *flist;
{
	int rfd, nw, nr, w, tobewritten;
	char buf[BUFSIZE], *bufp;

	if (copts->input_stdin) {
		rfd = 0;
	}
	else {
		if ((rfd = open(flist->filename, O_RDONLY, 0)) < 0) {
			perror("Open of input file failed\n");
			progexit(1);
		}
	}
	while (( nr = read(rfd, buf, BUFSIZE)) !=  0) {
		if (nr < 0) {
			perror("read of input file failed\n");
		    return(-1);	
		}	
		DBGPRT(("Numbytes Read from the Input file:: %d\n", nr))

		nw = 0;
		bufp = buf;
		tobewritten = nr;

		while (nw != nr) {
			if ((w = send(tpfd, bufp, tobewritten, 0)) < 0) {
				perror("send across tcp socket failed\n");
			    return(-1);	
			}
			nw +=w;
			bufp += w;
			tobewritten -= w;
			DBGPRT(("Numbytes sent to the printer:: %d\n", nw))
		}
	}
}

setup_udp()
{
	int ufd;
	struct sockaddr_in  claddr;
	if ((ufd = socket(PF_INET, SOCK_DGRAM, 0)) < 0) {
		perror("Failed to open UDP socket\n");
		progexit(1);
	}

	/*
	 * Bind any local address for us.
 	 */

	jzero((char *)&claddr, sizeof(claddr));
	claddr.sin_family = AF_INET;
	claddr.sin_port = htons(0) ;
	claddr.sin_addr.s_addr = htonl(INADDR_ANY);
        

	if (bind(ufd, (struct sockaddr_in *)&claddr, sizeof(claddr)) < 0){
		perror("UDP status bind failed\n");
		progexit(1);
	}
	
	return(ufd);
}

setup_backchannel()
{
	int ufd;
	struct sockaddr_in backaddr;

	if ((ufd = socket(PF_INET, SOCK_DGRAM, 0)) < 0) {
        	perror("Failed to open UDP socket for backchannel\n");
        	progexit(1);
        }
 
        /*
         * Bind any local address for us.
         */
 
        jzero((char *)&backaddr, sizeof(backaddr));
        backaddr.sin_family = AF_INET;
        backaddr.sin_port = htons(0) ;
        backaddr.sin_addr.s_addr = htonl(INADDR_ANY);
 
        if (bind(ufd, (struct sockaddr_in *)&backaddr, sizeof(backaddr))<0) {
		DBGPRT(("errno = %d\n\r",errno))
                perror("UDP sendback bind failed for backchannel\n");
                progexit(1);
        }
 
        return(ufd);
}

init_backchannel(ufd)
int ufd;
{
    int nfound, rcnt, recvlen, age, protocol_error, protocol_tries=1;
    char *strptr;
    char preply[256];
    struct sockaddr_in  recvaddr;
    struct printer_pkt  preq, *ps;
    char  reply[MAX_REPLY_SIZE];
    fd_set readfds;
    struct timeval timeout;


    timeout.tv_sec = 0;
    timeout.tv_usec = UDP_TIMEOUT * 10000;
    FD_ZERO (&readfds);
    FD_SET (ufd, &readfds);
    protocol_error = FALSE;

    while (!protocol_error) {
        while (TRUE) {
            nfound = select(ufd + 1, &readfds, (fd_set *)0, (fd_set *)0, &timeout);
            if ((nfound < 0) || (nfound > 1)) {
                perror("select failed\n");
                progexit(1);
            }
            else {
                if (nfound == 1) {
                    DBGPRT(("Found an old reply packet\n"))
                    FD_ZERO(&readfds);
                    FD_SET(ufd, &readfds);
                    if (recvfrom(ufd, &preply[0], sizeof(preply), 0,
                             (struct sockaddr *)&recvaddr, &recvlen) < 0) {
                        perror("Recvfrom:Failed to receive udp packet\n");
                        progexit(1);
 
                    }
                }
                else {
                    DBGPRT(("Timed out: No[more] past reply packets\n"))
                    break;
                }
            }
        }
        rcnt = NUMRETRY;
        while (rcnt--) {
            jzero((char *)&preq, sizeof(preq));
            if (bjobfile_present || blogfile_present) {
                preq.pr_spare = 2;
            }
            else {
                preq.pr_spare = 3;
            }

            if (sendto(ufd, (char *)&preq, sizeof(preq), 0, &printaddr,
                            sizeof(printaddr)) != sizeof(preq)) {
                perror("Sendto:Failed to send udp data printaddr\n");
                progexit(1);
            }

            /*
             * Issue a select and see if you get a packet within the timeout interval.
             */

            FD_ZERO (&readfds);
            FD_SET (ufd, &readfds);
            nfound = select(ufd +1, &readfds, (fd_set *)0, (fd_set *)0, &timeout);
            if ((nfound < 0) || (nfound > 1)) {
                perror("Select failed\n");
                progexit(1);
            }
            else {
                if (nfound == 0) {
                    DBGPRT(("Timed out: Reply packet not received from printer\n"))
                    continue;
                }
                else {
                    DBGPRT(("Reply packet received from printer\n"))
                }

            }

            /*
             * Read in the reply packet.
             */
            recvlen = sizeof(recvaddr);
            if (recvfrom(ufd, &preply[0], sizeof(preply), 0,
                         (struct sockaddr *)&recvaddr, &recvlen) < 0) {
                perror("Recvfrom:Failed to receive udp packet\n");
                progexit(1);
            }
            ps = (struct printer_pkt *)preply;
			DBGPRT(("Sent DB pr_spare=%d, received pr_spare=%d\n",preq.pr_spare,ps->pr_spare))
            if ((bjobfile_present || blogfile_present) &&
                                                 (ps->pr_spare == 2)) {
                protocol_error = TRUE;
            }
            else {
                if (ps->pr_spare == 3) {
                    protocol_error = TRUE;
                }
				else {
			        DBGPRT(("protocol_tries=%d\n",protocol_tries))
					if (protocol_tries == 2) {

		                if (bjobfile_present) {
                            fprintf(bjobptr, "Your printer firmware does not support backchannel message retrieval!\n");
                            fprintf(bjobptr, "(or the printer keypad Accnting/RevChan Mode is Disable)\n");
                            fflush(bjobptr);
                        }
		                if (blogfile_present) {
                            fprintf(blogptr, "Your printer firmware does not support backchannel message retrieval!\n");
                            fprintf(blogptr, "(or the printer keypad Accnting/RevChan Mode is Disable)\n");
                            fflush(blogptr);
                        }

						protocol_error =  TRUE;
						bjobfile_present = blogfile_present = FALSE;
                    }
		            else {
			            protocol_tries++;
                    }
                }
            }
            if (protocol_error) {
                return(1);	
            }
        }
        if (protocol_error) {
            return(0);
        }
    }
}

/*
 * Get printer status. This is an infinite loop; if no replies
 * are being received from the printer, the procedure keeps on sending
 * requests in the hope that a reply will be received. It would be trivial,
 * however, to add logic to terminate the procedure if no replies are
 * received after a long timeout, say 5 minutes.
 * Use of select() enables low level timeout mechanism to be enforced and so,
 * in theory, a network read, i.e. recvfrom(), should never block.
 * A reply packet from the printer indicates if the printer is busy or not.
 * If it is, the procedure, after RETRY counts, sleeps for SLEEPTIME before
 * sending the request packet again. The priming select() outside the 
 * inner while loop (RETRY loop), checks to see if any old packets have
 * finally found their destination and discards them.
 */
get_printstat(ufd, statflag, statbuf, term)
int ufd;
int statflag;
char *statbuf;
int term;
{
	int nfound, rcnt, rflags, pstatus, recvlen, age;
	char *strptr;
	char preply[256];
	struct sockaddr_in  recvaddr;
	struct printer_pkt  preq, *ps; 
	char  reply[MAX_REPLY_SIZE];
	fd_set readfds;
	struct timeval timeout;

	/*
	 * Set the time out value.
	 */
	timeout.tv_sec = 0;
	timeout.tv_usec = UDP_TIMEOUT * 10000;


		/* Flush out past reply packets, if any */
		FD_ZERO (&readfds);
		FD_SET (ufd, &readfds);

		while (TRUE) {
			nfound = select(ufd + 1, &readfds, (fd_set *)0, (fd_set *)0, &timeout);
			if ((nfound < 0) || (nfound > 1)) {
				perror("select failed\n");
				progexit(1);
			}
			else if (nfound == 1) {
				DBGPRT(("Found an old reply packet\n"))
				FD_ZERO(&readfds);
				FD_SET(ufd, &readfds);
				if (recvfrom(ufd, &preply[0], sizeof(preply), 0, 
						 (struct sockaddr *)&recvaddr, &recvlen) < 0) {
					perror("Recvfrom:Failed to receive udp packet\n");
					progexit(1);

				}
			}
			else {
				DBGPRT(("Timed out: No[more] past reply packets\n"))
				break;
			}
		}

		pstatus = 0;
		rcnt = NUMRETRY;
		while (rcnt--) {
			jzero((char *)&preq, sizeof(preq));
                        if (term) 
                            preq.pr_spare = 1;
			if (sendto(ufd, (char *)&preq, sizeof(preq), 0, &printaddr,
				sizeof(printaddr)) != sizeof(preq)) {
				perror("Sendto:Failed to send udp data \n");
				progexit(1);
			}

			/*
			 * Issue a select and see if you get a packet within the timeout interval.
			 */

			FD_ZERO (&readfds);
			FD_SET (ufd, &readfds);
			nfound = select(ufd +1, &readfds, (fd_set *)0, (fd_set *)0, &timeout);
			if ((nfound < 0) || (nfound > 1)) {
				perror("Select failed\n");
				progexit(1);
			}
			else if (nfound == 0) {
				DBGPRT(("Timed out: Reply packet not received from printer\n"))
				continue; 
			}
			else {
				DBGPRT(("Reply packet received from printer\n"))
			}

			/* 
			 * Read in the reply packet.
			 */
			recvlen = sizeof(recvaddr);
			if (recvfrom(ufd, &preply[0], sizeof(preply), 0, 
						 (struct sockaddr *)&recvaddr, &recvlen) < 0) {
				perror("Recvfrom:Failed to receive udp packet\n");
				progexit(1);
			}


			ps = (struct printer_pkt *)preply;

			DBGPRT(("Printer Status data:\n\t pr_status:0x%x\n\t pr_acceptmask:0x%x\n",
				ps->pr_status, ntohl(ps->pr_acceptmask)))

			pstatus = (!(ps->pr_status & 0x0f)) & (ntohl(ps->pr_acceptmask) & 0x01);
			DBGPRT(("print_status=%x\n", pstatus))

#ifdef DEBUG
			if (ps->pr_status & 0x02) 
				DBGPRT(("Printer lacks paper\n"))
			
			if (ps->pr_status & 0x04)
				DBGPRT(("Printer has paper jam\n"))
			
			if (ps->pr_status & 0x08)
				DBGPRT(("Printer lacks consumables (other than paper)\n"))

			if (ps->pr_status & 0x10)
				DBGPRT(("Job in progress at the printer\n"))

			if (pstatus != 0) {
				DBGPRT(("...printer ready to accept data\n"))
			}
			else {
				DBGPRT(("....printer busy or has problems\n"))
			}
#endif

			age = ntohl(ps->pr_age);

                        /*
                         * Copy the time from the age packet into the Time field
                         * of the accounting structure.
                         */

                        sprintf(account.Time, "%02d:%02d:%02d",
                                   age/3600, (age/60)%60, age%60);
                        if (ps->pr_status ==  0)
                            accounting.Terminate = 1;
                        else
                            accounting.Terminate = 0;

			strptr = (&preply[0] + ntohs(ps->pr_stroffset));
			/* Make sure string is null terminated */
			*(strptr + ntohs(ps->pr_strlength)) = 0x00;

			if (statflag) {
				sprintf(statbuf, "%s", strptr);
			}

			return(pstatus);

		} /* end rcnt while */

		DBGPRT(("....printer not responding \n"))

		if (statflag) {
			sprintf(statbuf, "Printer not responding to status request\n");
		}
		return(0);
}


init_data(copts)
struct command_opt *copts;
{
	copts->input_stdin = 1;
}

/*
 * Fork a child process to gather backchannel messages while a job is active.
 */
spawn_bchanproc(ufd)
int ufd;
{
    int pid;

    if ((pid = fork()) < 0) {
        printf("Failed to spawn backchannel process\n");
        exit(1);
    }
    if (pid == 0) { /* child */
        get_bchan_msg(ufd);
	}
    else {
        return(pid);
    }
}

/*
 * Fork a child process to gather printer status while a job is active.
 */

spawn_statproc()
{
	int pid, ufd;

	if ((pid = fork()) < 0) {
		printf("Failed to spawn status gathering process\n");
		exit(1);
	}
	if (pid == 0) { /* child */
		/*
	 	* Set up UDP socket.
	 	*/
		if ((ufd = setup_udp()) < 0) {
			printf("Failed to open UDP connection for child (status) process\n");
			exit(1);
		}

                /*
                 * if the orignal qef is desired call update_status otherwise
                 * use update_accounting.
                 */

                if (original)
                    update_status(ufd);
                else
                    update_accounting(ufd);
	}
	else {
		return(pid);
	}
}

update_status(ufd)
int ufd;
{
        int i;
        char statbuf[512], statstring[512];

        sleep(STATUSINTERVAL);

        i = 100;
        while (TRUE) {
                get_printstat(ufd, 1, statbuf);
                sprintf(statstring, "Job in progress--- Status: %s", statbuf);
                write_status(i++, statstring);
                sleep(STATUSINTERVAL);
        }
}

get_bchan_msg(ufd)
int ufd;
{
    int nfound, recvlen, i, j=0, packet_len;
    int pagecntr=0;
    struct sockaddr_in  recvaddr;
    char preply[MAX_REPLY_SIZE+1];
	char next_msg[MAX_REPLY_SIZE];
    fd_set readfds;
    struct timeval timeout;
    char *test1ptr, *test2ptr;
    char *cpbf;

	/*
	 * Set the time out value.
	 */
	if (copts.alternative) {
	    get_realtime();
	}

	timeout.tv_sec = 0;
	timeout.tv_usec = UDP_TIMEOUT * 10000;
	while (TRUE) {
		FD_ZERO (&readfds);
        FD_SET (ufd, &readfds);
        nfound = select(ufd +1, &readfds, (fd_set *)0, (fd_set *)0, &timeout);
        if ((nfound < 0) || (nfound > 1)) {
            perror("Select failed\n");
            progexit(1);
        }
        else if (nfound == 0) {
            DBGPRT(("Timed out: Backchannel packet not received from printer\n"))
			continue;
        }

        /*
         * Read in the reply packet.
        */
        recvlen = sizeof(recvaddr);
        if (recvfrom(ufd, &preply[0], sizeof(preply), 0,
            (struct sockaddr *)&recvaddr, &recvlen) < 0) {
            perror("Recvfrom:Failed to receive udp backchannel packet\n");
            progexit(1);
        }

        preply[MAX_REPLY_SIZE] = 0;
        if (uflag)
        if (strstr(preply, "status: ")){
            if (strstr(preply, "pagecount:"))
                if (!pagecntr){
                        cpbf = preply;
                        test1ptr=index(preply,':');
                        test1ptr++;
                        test2ptr=index(test1ptr,':');
                        test2ptr++;
                        sscanf(test2ptr,"%d",&postpagecnt);
                        pagecntr++;
                        j=0;
                        while (cpbf != test2ptr){
                                savebuf[j] = *cpbf;
                                j++;
                                cpbf++;
                        }
                        test1ptr=index(test2ptr,';');
                        sprintf(&savebuf[j]," 0%s",test1ptr);
                        j=0;
                        sprintf(preply,"%s",savebuf);
                }
                else{
                        continue;
                }
 
        }
        if (copts.alternative) {
            packet_len = strlen(preply);
            for (i=0;i<packet_len;i++) {
                next_msg[j++] = preply[i];
                if (preply[i] == '\n') {
                    next_msg[j] = 0;
                    if (bjobfile_present) {
                        fprintf(bjobptr, "%s%s",account.Time,next_msg);
                        fflush(bjobptr);
                    }
                    if (blogfile_present) {
                        fprintf(blogptr, "%s%s",account.Time,next_msg);
                        fflush(blogptr);
                    }
                    j = 0;
                }
            }
            if (j) {
                    next_msg[j] = 0;
                    if (bjobfile_present) {
                        fprintf(bjobptr, "%s%s\n",account.Time,next_msg);
                        fflush(bjobptr);
                    }
                    if (blogfile_present) {
                        fprintf(blogptr, "%s%s\n",account.Time,next_msg);
                        fflush(blogptr);
                    }
                    j = 0;
            }
               
        }
        else {
            if (bjobfile_present) {
                fprintf(bjobptr, "%s",preply);
                fflush(bjobptr);
            }
            if (blogfile_present) {
                fprintf(blogptr, "%s",preply);
                fflush(blogptr);
            }
        }
    }
}

get_realtime()
{ 

	int j, k, tmp;
	long clock;
        static char month[12][4]={"Jan","Feb","Mar","Apr",
                                  "May","Jun","Jul","Aug",
                                  "Sep","Oct","Nov","Dec"};
        struct time_info alttime;
        char temptime[7];

        /* 
         * Save the time.
         */
        clock = time((long *)0);
        strcpy(account.Time, ctime(&clock));
        tmp = strlen(account.Time);
        account.Time[tmp-1] = 0;
        if (copts.alternative) {
            sscanf(account.Time,"%s %s %s %s %s",
                   alttime.day,
                   alttime.month,
                   alttime.daynum,
                   alttime.hrminsec,
                   alttime.year);

            /* Get Month */
            for (j=1;j<14;j++) {
                   if (!strcmp(month[j-1],alttime.month)) {
                       if ((j-1)<10)
                           sprintf(account.Time,"0%d",j);
                       else sprintf(account.Time,"%d",j);
                       break;
                   }
            }
 
            /* Get Day of Month */
            if (j=(atoi(alttime.daynum)) < 10) {
                sprintf(temptime,"0%s",alttime.daynum);
                strcat(account.Time,temptime);
            }
            else {
               sprintf(temptime,"%s",alttime.daynum);
               strcat(account.Time,temptime);
            }
 
            /* Get Hours, Minutes, and Seconds */
            k=0;
            for (j=0;j<strlen(alttime.hrminsec);j++) {
               if (alttime.hrminsec[j] != ':')
                   temptime[k++] = alttime.hrminsec[j];
            }
            temptime[6] = 0;
            strcat(account.Time,temptime);
        }
 
}

update_accounting(ufd)
int ufd;
{

	int i, items_got=0, pos=0, tp, pstatus, length;
	char statstring[512];
        char old_statstring[512];
	char statbuf[512];
	char finalstat[525];
        char temp[160], temparray[512];
	char old_status[MODNAME_LEN];
        int indx_ptr, old_page, old_Recvd = 1;
	int count = 0;
	char *temp_ptr;
	int no_error_status = 0;

	sleep(copts.granularity);
        old_page = 0;
	i = 100;
        while (TRUE) {
                get_printstat(ufd, 1, statbuf, 0);
                strcpy(statstring, statbuf);
                temp_ptr = statstring;
 
                /*
                 * Get the User and Title info, if available
                 *
                 */
		for (i=0;temp_ptr[i];i++)	
		    if (temp_ptr[i] == '\n')
		    {
			no_error_status = 1;
			break;
		    }
	
		if (no_error_status)
		{
                	while (count < 7) {
                        	while (*temp_ptr++ != '\n');
                        	++count;
                	}
               		if (*temp_ptr == '\n')
			{
                       		strcpy(temp,++temp_ptr);
			}
               		else strcpy(temp, temp_ptr);
                	count = 0;
			no_error_status = 0;
		}
 
                /*
                 * Scan the remainder of statstring into
                 * the appropriate field of the structure.
                 */
                items_got = sscanf(statstring,"%d %d %d %d %d %d %d",
                                   &accounting.Status,
                                   &accounting.Page,
                                   &accounting.Recvd,
                                   &accounting.Sheet,
                                   &accounting.JobId,
                                   &accounting.ByteSize,
                                   &accounting.Emulation);
                 /*
                  * Process the user and job title info., if available.  The temp array will contain
                  * either the Title or the User name (or both), separated by a <LF>
                  */
 
                 if (strlen(temp) > 2){
                      indx_ptr = 0;
                      pos = 0;
                      while( temp[indx_ptr] != '\n')
                          temparray[pos++] = temp[indx_ptr++];
                      temparray[pos] = 0;
                      pos=0;
                      indx_ptr++;
                      if (temp[indx_ptr])       /* if there is no username, we will be at the end of statstring */
                                                /* after having just read in the title                          */
                      {
                        strcpy(accounting.For, temparray);
                        while(( temp[indx_ptr] != '\n')&&(indx_ptr <= strlen(temp)))
                                accounting.Title[pos++] = temp[indx_ptr++];
 
                        if (pos == 0) /* we have a username only */
                                sprintf(accounting.Title,"Untitled");
                        else
                                accounting.Title[pos] = 0;
                      }

		      else /* we have a title only */
		      {
			sprintf(accounting.For, "Unknown");
			strcpy(accounting.Title, temparray);
		      }
		      length = strlen(temp);
		      for (i=0;i<length;i++)
			   temp[i]=0;
                 }
                 else{ /* no username or title */ 
                     sprintf(accounting.For,"Unknown");
                     sprintf(accounting.Title,"Untitled");
                 }
		 

                 /*
                  * If real time is desired the copy in the
                  * System time of the message.
                  */
		 if (copts.time)
		     get_realtime();
                 
		 /*
   		  * Test the accounting.Status bit for printer status.
                  * Then copy equivalent string into account.Status.
                  */

                 if (items_got > 2){
                     if (accounting.Status & 0x20)
                        strcpy(account.Status,"waiting");
                     else
                     if (accounting.Status & 0x10)
		     {
			if (copts.alternative)
				strcpy(account.Status,"idle");
			else
				strcpy(account.Status,"terminating");
		     }
                     else
                     if (accounting.Status & 0x8)
		     {
			if (copts.alternative)
				strcpy(account.Status,"processing");
			else
		        	strcpy(account.Status,"printing");
		     }
                     else
                     if (accounting.Status & 0x4)
                        strcpy(account.Status,"rasterizing");
                     else
                     if (accounting.Status & 0x2)
		     {
			if (copts.alternative)
				strcpy(account.Status,"busy");
			else
                        	strcpy(account.Status,"interpreting");
		     }
                     else
                     if (accounting.Status & 0x1)
                        strcpy(account.Status,"spooling");

                     if (accounting.Recvd & 0x10 )
                        strcpy(account.Recvd,"Optional I/O");
                     else
                     if (accounting.Recvd == 13 )
                        strcpy(account.Recvd,"Parallel");
                     else
                     if (accounting.Recvd == 12 )
                        strcpy(account.Recvd,"Serial");
                     else
                        strcpy(account.Recvd,"Unknown");
                     /*
                      * Create the output string.
                      * Or copy in the error message.
                      */
		     if (copts.alternative)
			sprintf(statstring, "%s%%%%[status:%s; source:Ethernet:%s; pagecount:%d; user:%s; title:%s]%%%%",
				account.Time,
				account.Status,
				copts.hostname,	
				accounting.Sheet,
				accounting.For,
				accounting.Title);
		     else
                     	sprintf(statstring, "%%%%[status:%s; pagecount:%d; interface:%s:%s; user:%s; title:%s; time:%s]%%%%",  
                          	account.Status,
                          	accounting.Sheet,
                          	account.Recvd,
				copts.hostname,
                          	accounting.For,
                          	accounting.Title,
                          	account.Time);
                 }
                 else {
		      if (copts.alternative) {
			  if (strstr(statstring,"FEED JAM")||strstr(statstring,"INPUT BIN JAM"))
			      sprintf(statstring, "%%%%[PrinterError:paper entry misfeed]%%%%");
			  else if (strstr(statstring,"JAM"))
			           sprintf(statstring, "%%%%[PrinterError:paper jam]%%%%");
			       else if (strstr(statstring,"ADJUST INPUTBIN")||strstr(statstring,"ADJUST INPUT BIN"))
				        sprintf(statstring, "%%%%[PrinterError:no paper tray]%%%%");
				    else if(strstr(statstring,"LOW TONER"))
					     sprintf(statstring, "%%%%[PrinterError:toner low]%%%%");
					 else if(strstr(statstring,"TONER OUT"))
					          sprintf(statstring, "%%%%[PrinterError:out of toner]%%%%");
					      else if(strstr(statstring,"PUT")&&strstr(statstring,"PAPER"))
						       sprintf(statstring, "%%%%[PrinterError:out of paper]%%%%");
					      	   else sprintf(statstring, "%%%%[PrinterError:%s]%%%%",statbuf); 
		      }
		      else sprintf(statstring, "%%%%[PrinterError:%s]%%%%", statbuf);
                      if (strcmp(old_statstring, statstring) && !strstr(statstring,"IDLE") && 
				 !strstr(statstring,"NO ERROR")){
                          if (copts.alternative) {
                             sprintf(finalstat,"%s%s",account.Time,statstring);
                             write_accounting(1,finalstat);
                          } 
                          else
                             write_accounting(i++, statstring);
                      }
                      if (accounting.Terminate == 1)
                          exit(0);
                 }


                 /*
                  * We don't want to repeat old info... so only report
                  * info if the page changed or the status changed.  Also
                  * only report status pertaining to opt I/O.
                  */
                 if (verbose)
                     if ((accounting.Recvd & 0x10)&&
                        ((accounting.Page != old_page)||
                         (strcmp (account.Status,old_status) != 0) || 
			 (old_Recvd == 0)))
	                     write_accounting(i++, statstring);

		/*
                 * If the terminating message has been received then 
                 * write the last status message (non-verbose) and exit
                 */
 
                if ((items_got >= 1)&&(accounting.Recvd & 0x10))
		    if (accounting.Status & 0x10) {
                        if (!verbose)
                            write_accounting(i++, statstring);
                         pstatus = get_printstat(ufd, 1, statbuf, 1);
                /*
                 * If modified backchannel message needs to be sent to
                 * a file then do it.
                 */
                if (uflag)
                if (copts.alternative){
                        get_realtime();
                    if (bjobfile_present) {
                        fprintf(bjobptr, "%s%%%%[status: finished job; pagecount: %d;]%%%%\n",account.Time,accounting.Sheet);
                        fflush(bjobptr);
                    }
                    if (blogfile_present) {
                        fprintf(blogptr, "%s%%%%[status: finished job; pagecount: %d;]%%%%\n",account.Time,accounting.Sheet);
                        fflush(blogptr);
                    }
                }
                else
                {
                    if (bjobfile_present) {
                        fprintf(bjobptr, "%%%%[status: finished job; pagecount: %d;]%%%%\n",accounting.Sheet);
                        fflush(bjobptr);
                    }
                    if (blogfile_present) {
                        fprintf(blogptr, "%%%%[status: finished job; pagecount: %d;]%%%%\n",accounting.Sheet);
                        fflush(blogptr);
                    }
                }
                        exit(0);
                    }
               if (accounting.Terminate == 1)
                    exit(0);
                
                old_page = accounting.Page;
		old_Recvd = accounting.Recvd;
		strcpy (old_status,account.Status);
                sprintf(old_statstring, "%s",statstring);
		sleep(copts.granularity);
	}
}

/* 
 * Terminate the child  --- status gathering process.
 */
kill_statproc()
{
	if (kill(copts.cpid[0], SIGTERM) < 0) {
		if (errno != ESRCH) {
			perror("Failed to terminate the child (status gathering) process\n");
			exit(1);
		}
	}
#ifndef SGI
	while (wait(0) != copts.cpid[0]);
#endif
}

kill_bchanproc()
{
    if (kill(copts.cpid[1], SIGTERM) < 0) {
        if (errno != ESRCH) {
            perror("Failed to terminate the backchannel process(3)\n");
            exit(1);
        }
    }
#ifndef SGI
    while (wait(0) != copts.cpid[1]);
#endif
}

write_accounting(pstat, msg)
int pstat;
char *msg;
{
        /*
         * Write messages to all log files if they are being used
         */
	long clock;
	FILE *fptr;

	DBGPRT(("Status Message:: %s\n", msg))

	if (( fptr = fopen(copts.statfile, "w"))  ==  (FILE *)NULL) {
		perror("Failed to open status file\n");
		progexit(1);
	}

	clock = time((long *)0);

	fprintf(fptr, "%s  %s\n", ctime(&clock), msg);
	fflush(fptr);

        if (jobfile_present) {
            fprintf(jobptr, "%s\n", msg);
            fflush(jobptr);
        }

        if (logfile_present) {
            fprintf(logptr, "%s\n", msg);
            fflush(logptr);
        }
 
	if (fclose(fptr) < 0) {
		perror("Failed to close status file\n");
		progexit(1);
	}
}

write_status(pstat, msg)
int pstat;
char *msg;
{
        long clock;
        FILE *fptr;
        DBGPRT(("Status Message:: %s\n", msg))

        if (( fptr = fopen(copts.statfile, "w"))  ==  (FILE *)NULL) {
                perror("Failed to open status file\n");
                progexit(1);
        }

        clock = time((long *)0);

        fprintf(fptr, "%s  %s\n", ctime(&clock), msg);
        fflush(fptr);

        if (fclose(fptr) < 0) {
                perror("Failed to close status file\n");
                progexit(1);
        }
}

process_message(message, old_stat)
char *message;
char *old_stat;
{
int items_got;
char statstring[512];
char finalstat[525];

	                sprintf(statstring, "%s", message);

                /*
                 * Scan the string returned from the printer into
                 * the appropriate field of the structure.
                 */

                items_got = sscanf(statstring,"%d %d %d %d %d %d %d",
                                   &accounting.Status,
                                   &accounting.Page,
                                   &accounting.Recvd,
                                   &accounting.Page,
                                   &accounting.JobId,
                                   &accounting.ByteSize,
                                   &accounting.Emulation
                                   );
                      if (items_got < 2){
                          if (copts.alternative) {
                              if (strstr(statstring,"FEED JAM")||strstr(statstring,"INPUT BIN JAM"))
                                  sprintf(statstring, "%%%%[PrinterError:paper entry misfeed]%%%%");
                              else if (strstr(statstring,"JAM"))
                                       sprintf(statstring, "%%%%[PrinterError:paper jam]%%%%");
                                   else if (strstr(statstring,"ADJUST INPUTBIN")||strstr(statstring,"ADJUST INPUT BIN"))
                                            sprintf(statstring, "%%%%[PrinterError:no paper tray]%%%%");
                                        else if(strstr(statstring,"LOW TONER"))
                                                 sprintf(statstring, "%%%%[PrinterError:toner low]%%%%");
                                             else if(strstr(statstring,"TONER OUT"))
                                                      sprintf(statstring, "%%%%[PrinterError:out of toner]%%%%");
                                                  else if(strstr(statstring,"PUT")&&strstr(statstring,"PAPER"))
                                                           sprintf(statstring, "%%%%[PrinterError:out of paper]%%%%");
                                                       else sprintf(statstring, "%%%%[PrinterError:%s]%%%%",message);
                          }
                          else sprintf(statstring, "%%%%[PrinterError:%s]%%%%", message);
                          if (strcmp(old_stat, statstring) && !strstr(statstring,"IDLE") &&
                                     !strstr(statstring,"NO ERROR")){
		              sprintf(old_stat,"%s",statstring);	
			      if (copts.alternative) {
			         get_realtime();
				 sprintf(finalstat,"%s%s",account.Time,statstring); 
			         write_accounting(1,finalstat);
			      }
			      else
                                 write_accounting(1, statstring);
                          }
                      }
}
progexit(exitcode)
int exitcode;
{
	if (copts.cpid[0] != getpid()) { /* not child process -- parent process */
		kill_statproc();
	}
	if (copts.cpid[1] != getpid()) { 
		kill_bchanproc();
	}
 	exit(exitcode);		
}
